<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

function elyssa_ai_update_debug_page() {
    // Nur für Administratoren
    if (!current_user_can('manage_options')) {
        wp_die(__('You do not have sufficient permissions to access this page.', 'elyssa-ai'));
    }

    // Wenn der Reset-Button geklickt wurde
    if (isset($_POST['elyssa_reset_transients']) && wp_verify_nonce($_POST['elyssa_update_debug_nonce'], 'elyssa_update_debug')) {
        // Transients zurücksetzen
        delete_transient('elyssa_ai_update_check');
        delete_transient('elyssa_ai_plugin_info');
        delete_site_transient('update_plugins');
        wp_clean_plugins_cache(true);
        wp_update_plugins();
        
        // Nutzer-Feedback
        echo '<div class="notice notice-success is-dismissible"><p>';
        _e('Transients have been reset and update check forced.', 'elyssa-ai');
        echo '</p></div>';
    }

    // Wenn der Check-Button geklickt wurde
    if (isset($_POST['elyssa_check_updates']) && wp_verify_nonce($_POST['elyssa_update_debug_nonce'], 'elyssa_update_debug')) {
        // Updater initialisieren und manuellen Check durchführen
        require_once ELYSSA_AI_PLUGIN_DIR . 'updates/elyssa-updater.php';
        $updater = new ElyssaAI_Updater();
        $update_info = $updater->force_update_check();
        
        // Nutzer-Feedback
        if (is_wp_error($update_info)) {
            echo '<div class="notice notice-error is-dismissible"><p>';
            echo sprintf(__('Error checking for updates: %s', 'elyssa-ai'), $update_info->get_error_message());
            echo '</p></div>';
        } else if (isset($update_info->version)) {
            if (version_compare(ELYSSA_AI_VERSION, $update_info->version, '<')) {
                echo '<div class="notice notice-info is-dismissible"><p>';
                echo sprintf(__('Update available! Version %s is available. You are running version %s.', 'elyssa-ai'), 
                    esc_html($update_info->version), esc_html(ELYSSA_AI_VERSION));
                echo '</p></div>';
            } else {
                echo '<div class="notice notice-success is-dismissible"><p>';
                echo __('You have the latest version installed.', 'elyssa-ai');
                echo '</p></div>';
            }
        } else {
            echo '<div class="notice notice-warning is-dismissible"><p>';
            echo __('Received invalid update information from the server.', 'elyssa-ai');
            echo '</p></div>';
        }
    }
    
    // Plugin-Information sammeln
    $plugin_basename = plugin_basename(ELYSSA_AI_PLUGIN_FILE);
    $plugin_slug = dirname($plugin_basename);
    
    // Aktuelle WordPress-Transients
    $update_plugins = get_site_transient('update_plugins');
    
    // Spezifische Plugin-Transients
    $elyssa_transient = get_transient('elyssa_ai_update_check');
    $elyssa_plugin_info = get_transient('elyssa_ai_plugin_info');

    // Debug-Informationen anzeigen
    ?>
    <div class="wrap">
        <h1><?php esc_html_e('Elyssa AI Update Debug', 'elyssa-ai'); ?></h1>
        
        <div class="card">
            <h2><?php esc_html_e('Plugin Information', 'elyssa-ai'); ?></h2>
            <table class="widefat">
                <tr>
                    <th><?php esc_html_e('Current Version', 'elyssa-ai'); ?></th>
                    <td><?php echo esc_html(ELYSSA_AI_VERSION); ?></td>
                </tr>
                <tr>
                    <th><?php esc_html_e('Plugin Basename', 'elyssa-ai'); ?></th>
                    <td><?php echo esc_html($plugin_basename); ?></td>
                </tr>
                <tr>
                    <th><?php esc_html_e('Plugin Slug', 'elyssa-ai'); ?></th>
                    <td><?php echo esc_html($plugin_slug); ?></td>
                </tr>
                <tr>
                    <th><?php esc_html_e('Update Server URL', 'elyssa-ai'); ?></th>
                    <td>https://get.codefabrica.net/wp/elyssa-ai/info</td>
                </tr>
            </table>
        </div>
        
        <div class="card" style="margin-top: 20px;">
            <h2><?php esc_html_e('Update Actions', 'elyssa-ai'); ?></h2>
            <form method="post" action="">
                <?php wp_nonce_field('elyssa_update_debug', 'elyssa_update_debug_nonce'); ?>
                <p>
                    <input type="submit" name="elyssa_check_updates" class="button button-primary" value="<?php esc_attr_e('Check for Updates', 'elyssa-ai'); ?>" />
                    <input type="submit" name="elyssa_reset_transients" class="button button-secondary" value="<?php esc_attr_e('Reset Update Transients', 'elyssa-ai'); ?>" />
                </p>
            </form>
        </div>
        
        <div class="card" style="margin-top: 20px;">
            <h2><?php esc_html_e('Current Transient Status', 'elyssa-ai'); ?></h2>
            <table class="widefat">
                <tr>
                    <th colspan="2"><?php esc_html_e('WordPress Update Plugins Transient', 'elyssa-ai'); ?></th>
                </tr>
                <tr>
                    <th><?php esc_html_e('Has Checked Data', 'elyssa-ai'); ?></th>
                    <td><?php echo isset($update_plugins->checked) ? '✓' : '✗'; ?></td>
                </tr>
                <tr>
                    <th><?php esc_html_e('Has Response Data', 'elyssa-ai'); ?></th>
                    <td><?php echo isset($update_plugins->response) ? '✓' : '✗'; ?></td>
                </tr>
                <tr>
                    <th><?php esc_html_e('Our Plugin in Checked', 'elyssa-ai'); ?></th>
                    <td>
                        <?php 
                        if (isset($update_plugins->checked[$plugin_basename])) {
                            echo '✓ Version: ' . esc_html($update_plugins->checked[$plugin_basename]);
                        } else {
                            echo '✗ Not found';
                        }
                        ?>
                    </td>
                </tr>
                <tr>
                    <th><?php esc_html_e('Our Plugin in Response', 'elyssa-ai'); ?></th>
                    <td>
                        <?php 
                        if (isset($update_plugins->response[$plugin_basename])) {
                            $response = $update_plugins->response[$plugin_basename];
                            echo '✓ New version: ' . esc_html($response->new_version);
                        } else {
                            echo '✗ No update available or not checked';
                        }
                        ?>
                    </td>
                </tr>
            </table>
        </div>
        
        <div class="card" style="margin-top: 20px;">
            <h2><?php esc_html_e('Test Update Server Connection', 'elyssa-ai'); ?></h2>
            <pre id="elyssa-server-response" style="background: #f8f8f8; padding: 10px; overflow: auto; max-height: 400px;">
                <?php
                // Die Anfrage direkt durchführen
                $params = array(
                    'action' => 'plugin_information',
                    'slug' => $plugin_slug,
                    'version' => ELYSSA_AI_VERSION
                );
                
                $request_url = add_query_arg($params, 'https://get.codefabrica.net/wp/elyssa-ai/info');
                
                $args = array(
                    'timeout' => 15,
                    'sslverify' => false
                );
                
                $response = wp_remote_get($request_url, $args);
                
                if (is_wp_error($response)) {
                    echo "Error: " . esc_html($response->get_error_message());
                } else {
                    $response_code = wp_remote_retrieve_response_code($response);
                    $body = wp_remote_retrieve_body($response);
                    
                    echo "Response Code: " . esc_html($response_code) . "\n\n";
                    echo "Body:\n" . esc_html($body);
                    
                    // JSON dekodieren und Version vergleichen
                    $data = json_decode($body);
                    if ($data && isset($data->version)) {
                        echo "\n\nVersion comparison:\n";
                        echo "Server version: " . esc_html($data->version) . "\n";
                        echo "Local version: " . esc_html(ELYSSA_AI_VERSION) . "\n";
                        echo "Result: " . (version_compare(ELYSSA_AI_VERSION, $data->version, '<') ? "Update available" : "Up to date");
                    }
                }
                ?>
            </pre>
        </div>
    </div>
    <?php
}

// Admin-Menüeintrag für die Debug-Seite hinzufügen, aber nur für Administratoren und wenn Logging aktiviert ist
function elyssa_ai_add_update_debug_menu() {
    if (current_user_can('manage_options') && get_option('elyssa_ai_logging_enabled')) {
        add_submenu_page(
            'elyssa-ai',
            __('Update Debug', 'elyssa-ai'),
            __('Update Debug', 'elyssa-ai'),
            'manage_options',
            'elyssa-ai-update-debug',
            'elyssa_ai_update_debug_page'
        );
    }
}
add_action('admin_menu', 'elyssa_ai_add_update_debug_menu', 99);
