<?php
if (!defined('ABSPATH')) {
    exit;
}

class ElyssaAIChat extends ElyssaAIBaseService
{
    public function __construct() {
        add_action('admin_bar_menu', array($this, 'add_chat_menu_item'), 999);
        add_action('admin_footer', array($this, 'add_chat_modal'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_chat_scripts'));
        add_action('wp_ajax_elyssa_ai_chat_message', array($this, 'handle_chat_message'));
        add_action('wp_ajax_elyssa_ai_save_chat', array($this, 'save_chat'));
        add_action('wp_ajax_elyssa_ai_load_chat', array($this, 'load_chat'));
        add_action('wp_ajax_elyssa_ai_delete_chat', array($this, 'delete_chat'));
    }

    public function add_chat_menu_item($admin_bar) {
        $admin_bar->add_menu(array(
            'id'    => 'elyssa-ai-chat',
            'title' => 'elyssa AI Chat',
            'href'  => '#',
            'meta'  => array(
                'onclick' => 'elyssaAIOpenChat(); return false;',
            ),
        ));
    }

    public function add_chat_modal() {
        // Anstatt direktes HTML auszugeben, erstellen wir eine leere Div-Container
        // und fügen den Modal-Inhalt über JavaScript hinzu
        add_action('admin_footer', function() {
            echo '<div id="elyssa-ai-chat-container"></div>';
        });
        
        // Chat-Modal-Markup-Generierung in die JS-Datei verschieben
        // Die Daten werden über wp_localize_script in enqueue_chat_scripts() übergeben
    }

    public function enqueue_chat_scripts() {
        wp_enqueue_script('elyssa-ai-chat-js', plugin_dir_url(__FILE__) . 'assets/elyssa-ai-chat.js?'.time(), array('jquery'), ELYSSA_AI_VERSION, true);
        wp_enqueue_style('elyssa-ai-chat-css', plugin_dir_url(__FILE__) . 'assets/elyssa-ai-chat.css?'.time(), array(), ELYSSA_AI_VERSION);
        wp_enqueue_script('wp-i18n');
        wp_set_script_translations('elyssa-ai-chat-js', 'elyssa-ai');

        $provider = get_option('elyssa_ai_text_service', '');
        $model = get_option('elyssa_ai_text_model', '');

        wp_localize_script('elyssa-ai-chat-js', 'elyssaAiChatVars', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('elyssa_ai_chat_nonce'),
            'provider' => $provider,
            'model' => $model,
            'isTextEnabled' => elyssa_is_text_enabled(),
            'logoUrl' => esc_url(plugins_url('elyssa-ai/images/elyssa-logo-h100.webp')),
            'sendText' => esc_html__('Send', 'elyssa-ai'),
            'serviceDisabledText' => esc_html__('Please enable text services in the settings', 'elyssa-ai')
        ));
    }

    public function handle_chat_message() {
        check_ajax_referer('elyssa_ai_chat_nonce', 'nonce');

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'elyssa-ai')));
            return;
        }

        $message = isset($_POST['message']) ? sanitize_textarea_field(wp_unslash($_POST['message'] )) : '';
        $context = isset($_POST['context']) ? wp_kses_post(wp_unslash($_POST['context'])) : '';

        if (empty($message)) {
            wp_send_json_error(array('message' => __('No message provided', 'elyssa-ai')));
            return;
        }

        $init_result = $this->init_text_service();
        if (is_wp_error($init_result)) {
            wp_send_json_error(array('message' => $init_result->get_error_message()));
            return;
        }

        try {
            $full_message = $context . "\n\nHuman: " . $message . "\n\nAssistant:";
            $response = $this->text_service->generate_text($full_message);
            $provider = get_option('elyssa_ai_text_service', '');
            $model = get_option('elyssa_ai_text_model', '');

            wp_send_json_success(array(
                'message' => $response,
                'provider' => $provider,
                'model' => $model
            ));
        } catch (Exception $e) {
            wp_send_json_error(array('message' => $e->getMessage()));
        }
    }

    public function save_chat() {
        check_ajax_referer('elyssa_ai_chat_nonce', 'nonce');

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'elyssa-ai')));
            return;
        }

        $chat_history = isset($_POST['chat_history']) ? wp_kses_post(wp_unslash($_POST['chat_history'])) : '';
        $user_id = get_current_user_id();

        update_user_meta($user_id, 'elyssa_ai_chat_history', $chat_history);

        wp_send_json_success(array('message' => __('Chat saved successfully', 'elyssa-ai')));
    }

    public function load_chat() {
        check_ajax_referer('elyssa_ai_chat_nonce', 'nonce');

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'elyssa-ai')));
            return;
        }

        $user_id = get_current_user_id();
        $chat_history = get_user_meta($user_id, 'elyssa_ai_chat_history', true);

        wp_send_json_success(array('chat_history' => $chat_history));
    }

    public function delete_chat() {
        check_ajax_referer('elyssa_ai_chat_nonce', 'nonce');

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'elyssa-ai')));
            return;
        }

        $user_id = get_current_user_id();
        delete_user_meta($user_id, 'elyssa_ai_chat_history');

        wp_send_json_success(array('message' => __('Chat history deleted successfully', 'elyssa-ai')));
    }
}
new ElyssaAIChat();