document.addEventListener('DOMContentLoaded', function() {
    console.log(wp.i18n.__('elyssa AI Classic Editor script started', 'elyssa-ai'));

    // Helper function for AJAX requests
    function elyssaAjaxRequest(action, data) {
        showLoadingIndicator();

        const formData = new FormData();
        formData.append('action', 'elyssa_ai_' + action);
        formData.append('elyssa_ai_editor_nonce', elyssaAiEditorVars.nonce);
        formData.append('nonce', elyssaAiEditorVars.nonce);

        for (const key in data) {
            formData.append(key, data[key]);
        }

        return fetch(elyssaAiEditorVars.ajax_url, {
            method: 'POST',
            credentials: 'same-origin',
            body: formData
        })
            .then(response => {
                if (!response.ok) {
                    throw new Error('Network response was not ok');
                }
                return response.json();
            })
            .finally(hideLoadingIndicator);
    }

    // Loading indicator functions
    function showLoadingIndicator() {
        if (!document.getElementById('elyssaLoadingIndicator')) {
            const loadingDiv = document.createElement('div');
            loadingDiv.id = 'elyssaLoadingIndicator';
            loadingDiv.textContent = wp.i18n.__('Loading...', 'elyssa-ai');
            loadingDiv.style.cssText = 'position: fixed; top: 0; left: 0; width: 100%; height: 100%; background-color: rgba(0, 0, 0, 0.5); color: white; font-size: 24px; display: flex; justify-content: center; align-items: center; z-index: 9999;';
            document.body.appendChild(loadingDiv);
        } else {
            document.getElementById('elyssaLoadingIndicator').style.display = 'flex';
        }
    }

    function hideLoadingIndicator() {
        const loadingDiv = document.getElementById('elyssaLoadingIndicator');
        if (loadingDiv) {
            loadingDiv.style.display = 'none';
        }
    }

    // Get elements safely
    function getElement(id) {
        return document.getElementById(id);
    }

    // Add event listener safely
    function addListener(elementId, event, handler) {
        const element = getElement(elementId);
        if (element) {
            element.addEventListener(event, handler);
        } else {
            console.warn(`Element with id '${elementId}' not found`);
        }
    }

    // Check if we're on the correct page
    const content = getElement('content');
    if (!content) {
        console.log('Not on a post/page edit screen, skipping editor initialization');
        return;
    }

    function updateEditorContent(content) {
        console.log('Updating editor content:', content);
        const contentElement = getElement('content');
        if (contentElement) {
            contentElement.value = content;
        }

        // If TinyMCE is active, update its content as well
        if (typeof tinyMCE !== 'undefined' && tinyMCE.activeEditor && !tinyMCE.activeEditor.isHidden()) {
            tinyMCE.activeEditor.setContent(content);
        }
    }

    function rewriteContent() {
        console.log(wp.i18n.__('Rewrite content clicked', 'elyssa-ai'));
        const contentElement = getElement('content');
        if (!contentElement) {
            console.error('Content element not found');
            return;
        }

        elyssaAjaxRequest('rewrite', {content: contentElement.value})
            .then(response => {
                console.log('Rewrite response:', response);
                if (response.success && response.data) {
                    const titleElement = getElement('title');
                    if (response.data.title && titleElement) {
                        titleElement.value = response.data.title;
                    }
                    if (response.data.content) {
                        updateEditorContent(response.data.content);
                    }
                    if (response.data.keywords) {
                        updateKeywords(response.data.keywords);
                    }
                } else {
                    throw new Error(response.data ? response.data.message : 'Unknown error');
                }
            })
            .catch(error => {
                console.error('Rewrite error:', error);
                alert('An error occurred while processing your request: ' + error.message);
            });
    }

    function translateContent() {
        console.log(wp.i18n.__('Translate content clicked', 'elyssa-ai'));
        const contentElement = getElement('content');
        const langSelect = getElement('elyssa-ai-translate-language');

        if (!contentElement || !langSelect) {
            console.error('Required elements not found');
            return;
        }

        const lang = langSelect.value;
        if (!lang) {
            alert(wp.i18n.__('Please select a language', 'elyssa-ai'));
            return;
        }

        elyssaAjaxRequest('translate', {content: contentElement.value, language: lang})
            .then(response => {
                console.log('Translate response:', response);
                if (response.success && response.data) {
                    const titleElement = getElement('title');
                    if (response.data.title && titleElement) {
                        titleElement.value = response.data.title;
                    }
                    if (response.data.content) {
                        updateEditorContent(response.data.content);
                    }
                    if (response.data.keywords) {
                        updateKeywords(response.data.keywords);
                    }
                } else {
                    throw new Error(response.data ? response.data.message : 'Unknown error');
                }
            })
            .catch(error => {
                console.error('Translate error:', error);
                alert('An error occurred while processing your request: ' + error.message);
            });
    }

    function createNewArticle() {
        const promptElement = getElement('elyssa-ai-new-article-prompt');
        if (!promptElement) {
            console.error('Prompt element not found');
            return;
        }

        const prompt = promptElement.value;
        if (!prompt) {
            alert(wp.i18n.__('Please enter a prompt', 'elyssa-ai'));
            return;
        }

        elyssaAjaxRequest('new_article', {prompt: prompt})
            .then(response => {
                console.log('New article response:', response);
                if (response.success && response.data) {
                    const titleElement = getElement('title');
                    if (response.data.title && titleElement) {
                        titleElement.value = response.data.title;
                    }
                    if (response.data.content) {
                        updateEditorContent(response.data.content);
                    }
                    if (response.data.keywords) {
                        updateKeywords(response.data.keywords);
                    }
                } else {
                    throw new Error(response.data ? response.data.message : 'Unknown error');
                }
            })
            .catch(error => {
                console.error('New article error:', error);
                alert('An error occurred while processing your request: ' + error.message);
            });
    }

    function getPromptContent(promptId, targetSelector) {
        elyssaAjaxRequest('get_prompt_content', {prompt_id: promptId})
            .then(response => {
                const targetElement = document.querySelector(targetSelector);
                if (!targetElement) {
                    console.error('Target element not found:', targetSelector);
                    return;
                }

                if (response.success && response.data && response.data.user_prompt) {
                    targetElement.value = response.data.user_prompt;
                } else {
                    throw new Error(response.data ? response.data.message : 'Unknown error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('An error occurred while fetching the prompt content: ' + error.message);
            });
    }

    function generateKeywords() {
        console.log(wp.i18n.__('Generate keywords clicked', 'elyssa-ai'));
        const contentElement = getElement('content');
        if (!contentElement) {
            console.error('Content element not found');
            return;
        }

        elyssaAjaxRequest('generate_keywords', {content: contentElement.value})
            .then(response => {
                console.log('Generate keywords response:', response);
                if (response.success && response.data && response.data.keywords) {
                    updateKeywords(response.data.keywords);
                    alert(wp.i18n.__('Keywords have been generated and added to the post tags.', 'elyssa-ai'));
                } else {
                    throw new Error(response.data ? response.data.message : wp.i18n.__('No keywords were generated.', 'elyssa-ai'));
                }
            })
            .catch(error => {
                console.error('Generate keywords error:', error);
                alert(wp.i18n.__('An error occurred while processing your request: ', 'elyssa-ai') + error.message);
            });
    }

    function updateKeywords(keywords) {
        if (Array.isArray(keywords)) {
            keywords = keywords.join(', ');
        }

        const tagInput = getElement('new-tag-post_tag');
        if (!tagInput) {
            console.error('Tag input element not found');
            return;
        }

        const currentTags = tagInput.value;

        // Add new keywords to the input field
        if (currentTags) {
            tagInput.value = currentTags + ', ' + keywords;
        } else {
            tagInput.value = keywords;
        }

        // Trigger the addition of tags
        if (typeof tagBox !== 'undefined') {
            tagBox.flushTags(getElement('post_tag'), false, 1);
        }

        // Clear the tag input field
        tagInput.value = '';
    }

    function generateMedia() {
        console.log(wp.i18n.__('Generate media clicked', 'elyssa-ai'));
        const promptElement = getElement('elyssa-ai-image-prompt');
        if (!promptElement) {
            console.error('Image prompt element not found');
            return;
        }

        const prompt = promptElement.value;
        if (!prompt) {
            alert(wp.i18n.__('Please enter an image prompt', 'elyssa-ai'));
            return;
        }

        elyssaAjaxRequest('generate_media', {prompt: prompt})
            .then(response => {
                console.log('Generate media response:', response);
                if (response.success && response.data && response.data.url) {
                    insertMediaToEditor(response.data.url);
                } else {
                    throw new Error(response.data ? response.data.message : 'Unknown error');
                }
            })
            .catch(error => {
                console.error('Generate media error:', error);
                alert('An error occurred while processing your request: ' + error.message);
            });
    }

    function insertMediaToEditor(url) {
        const content = '<img src="' + url + '" alt="' + wp.i18n.__('Generated Image', 'elyssa-ai') + '">';
        const editor = tinyMCE.activeEditor;

        if (editor && !editor.isHidden()) {
            editor.execCommand('mceInsertContent', false, content);
        } else {
            const textarea = getElement('content');
            if (textarea) {
                const currentContent = textarea.value;
                textarea.value = currentContent + content;
            }
        }
    }

    // Initialize event listeners
    function initializeEventListeners() {
        // Core editor functions
        addListener('elyssa-ai-rewrite', 'click', rewriteContent);
        addListener('elyssa-ai-translate', 'click', translateContent);
        addListener('elyssa-ai-new-article', 'click', createNewArticle);
        addListener('elyssa-ai-generate-media', 'click', generateMedia);
        addListener('elyssa-ai-generate-keywords', 'click', generateKeywords);

        // Prompt selectors
        const textPromptSelect = getElement('elyssa-ai-text-prompt-select');
        if (textPromptSelect) {
            textPromptSelect.addEventListener('change', function() {
                const promptId = this.value;
                if (promptId) {
                    getPromptContent(promptId, '#elyssa-ai-new-article-prompt');
                }
            });
        }

        const imagePromptSelect = getElement('elyssa-ai-image-prompt-select');
        if (imagePromptSelect) {
            imagePromptSelect.addEventListener('change', function() {
                const promptId = this.value;
                if (promptId) {
                    getPromptContent(promptId, '#elyssa-ai-image-prompt');
                }
            });
        }
    }

    // Initialize the editor
    console.log(wp.i18n.__('Document ready, attaching event listeners', 'elyssa-ai'));
    initializeEventListeners();
});