<?php
if (!defined('ABSPATH')) {
    exit;
}

wp_enqueue_script( 'wp-i18n' );

add_action( 'wp_ajax_elyssa_ai_create_post', 'elyssa_ai_create_post_handler' );

function elyssa_ai_create_post_handler() {
    check_ajax_referer( 'elyssa_ai_create_post_action', 'elyssa_ai_create_post_nonce' );

    if (!current_user_can('edit_posts')) {
        wp_send_json_error(array('message' => __('Insufficient permissions', 'elyssa-ai')));
        return;
    }

    $title = isset($_POST['title']) ? sanitize_text_field(wp_unslash($_POST['title'])) : '';
    $content = isset($_POST['content']) ? wp_kses_post(wp_unslash($_POST['content'])) : '';
    $tags = isset($_POST['tags']) ? sanitize_text_field(wp_unslash($_POST['tags'])) : '';

    $post_data = array(
        'post_title'   => $title,
        'post_content' => $content,
        'post_status'  => 'draft',
        'post_author'  => get_current_user_id(),
        'post_type'    => 'post',
    );

    $post_id = wp_insert_post($post_data);

    if ( $post_id ) {
        $image_url = !empty($_POST['image']) ? sanitize_text_field(wp_unslash($_POST['image'])) : 'off';

        if ( $image_url !== 'off' ) {
            $attach_id = elyssa_ai_download_image_to_media_library( $image_url, $post_id );

            if ( $attach_id ) {
                elyssa_ai_set_featured_image( $post_id, $attach_id );
            }
        }

        if (!empty($tags)) {
            $tag_array = explode(',', $tags);
            $tag_array = array_map('trim', $tag_array);
            wp_set_post_tags($post_id, $tag_array);
        }

        wp_send_json_success( [ 'post_id' => $post_id ] );
    } else {
        wp_send_json_error();
    }

    wp_die();
}

add_action( 'wp_ajax_elyssa_ai_create_page', 'elyssa_ai_create_page_handler' );

function elyssa_ai_create_page_handler() {
    check_ajax_referer( 'elyssa_ai_create_page_action', 'elyssa_ai_create_page_nonce' );

    if (!current_user_can('edit_posts')) {
        wp_send_json_error(array('message' => __('Insufficient permissions', 'elyssa-ai')));
        return;
    }

    $title = isset($_POST['title']) ? sanitize_text_field(wp_unslash($_POST['title'])) : '';
    $content = isset($_POST['content']) ? wp_kses_post(wp_unslash($_POST['content'])) : '';
    $tags = isset($_POST['tags']) ? sanitize_text_field(wp_unslash($_POST['tags'])) : '';

    $post_data = array(
        'post_title'   => $title,
        'post_content' => $content,
        'post_status'  => 'draft',
        'post_author'  => get_current_user_id(),
        'post_type'    => 'page',
    );

    $post_id = wp_insert_post( $post_data );

    if ( $post_id ) {
        $image_url = !empty($_POST['image']) ? sanitize_text_field(wp_unslash($_POST['image'])) : 'off';
        if ( $image_url !== 'off' ) {
            $attach_id = elyssa_ai_download_image_to_media_library( $image_url, $post_id );

            if ( $attach_id ) {
                elyssa_ai_set_featured_image( $post_id, $attach_id );
            }
        }

        if (!empty($tags)) {
            $tag_array = explode(',', $tags);
            $tag_array = array_map('trim', $tag_array);
            wp_set_post_tags($post_id, $tag_array);
        }

        wp_send_json_success( [ 'post_id' => $post_id ] );
    } else {
        wp_send_json_error();
    }

    wp_die();
}

function elyssa_ai_download_image_to_media_library($image_url, $post_id = 0, $description = '') {
    require_once(ABSPATH . 'wp-admin/includes/image.php');
    require_once(ABSPATH . 'wp-admin/includes/file.php');
    require_once(ABSPATH . 'wp-admin/includes/media.php');

    elyssa_ai_log(__("Attempting to download image: ", 'elyssa-ai') . $image_url);

    // Download file to temp dir
    $tmp = download_url($image_url);
    if (is_wp_error($tmp)) {
        elyssa_ai_log(__("Error downloading image: ", 'elyssa-ai') . $tmp->get_error_message());
        return false;
    }

    elyssa_ai_log(__("Image downloaded to temp file: ", 'elyssa-ai') . $tmp);

    // Clean up the file name
    $url_parts = wp_parse_url($image_url);
    $clean_path = pathinfo($url_parts['path']);
    $file_name = $clean_path['filename'] . '.' . $clean_path['extension'];

    $file_array = array(
        'name' => $file_name,
        'tmp_name' => $tmp
    );

    // Check for valid image
    $file_type = wp_check_filetype($file_name);
    if (!$file_type['type']) {
        elyssa_ai_log(__("Invalid file type for image: ", 'elyssa-ai') . $file_name);
        wp_delete_file($tmp);
        return false;
    }

    elyssa_ai_log(__("Valid file type: ", 'elyssa-ai') . $file_type['type']);

    // Upload the image
    $attachment_id = media_handle_sideload($file_array, $post_id);

    if (is_wp_error($attachment_id)) {
        elyssa_ai_log(__("Error uploading image: ", 'elyssa-ai') . $attachment_id->get_error_message());
        wp_delete_file($tmp);
        return false;
    }

    elyssa_ai_log(__("Image uploaded successfully. Attachment ID: ", 'elyssa-ai') . $attachment_id);

    // If a description was provided, update the attachment post
    if (!empty($description)) {
        $update_result = wp_update_post(array(
            'ID' => $attachment_id,
            'post_excerpt' => $description // This sets the caption
        ));
        if (is_wp_error($update_result)) {
            elyssa_ai_log(__("Error updating attachment description: ", 'elyssa-ai') . $update_result->get_error_message());
        } else {
            elyssa_ai_log(__("Attachment description updated successfully", 'elyssa-ai'));
        }
    }

    return $attachment_id;
}

function elyssa_ai_set_featured_image( $post_id, $attach_id ) {
    set_post_thumbnail( $post_id, $attach_id );
}

function elyssa_ai_enqueue_styles($hook) {
    wp_enqueue_style('elyssa-ai-css', plugin_dir_url(__FILE__) . '../editor/css/elyssa-ai.css', array(), '1.0');
}
add_action('admin_enqueue_scripts', 'elyssa_ai_enqueue_styles');

function elyssa_ai_enqueue_tailwind_js() {
    // Check if the GET parameter 'page' is set and corresponds to 'elyssa-ai'
    $page = sanitize_text_field(wp_unslash($_GET['page'] ?? ''));
    $request = sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'] ?? ''));
    if ($page && substr($page,0,9) === 'elyssa-ai' ||
        basename($request) == 'upload.php') {
        // Path to Tailwind JavaScript file
        $tailwind_js_url = 'https://cdn.tailwindcss.com';

        // Enqueue Tailwind JavaScript
        wp_enqueue_script('elyssa-ai-tailwind-js', $tailwind_js_url, array(), '1.0.0', true);
    }
}
add_action('admin_enqueue_scripts', 'elyssa_ai_enqueue_tailwind_js', 999);

function elyssa_ai_add_inline_script() {
    $inline_script = 'tailwind.config = {corePlugins: {preflight: false}}';
    wp_add_inline_script('elyssa-ai-script', $inline_script);
}
add_action('wp_enqueue_scripts', 'elyssa_ai_add_inline_script');

function elyssa_ai_enqueue_scripts($hook) {
    // Make sure the script is only loaded on the edit page
    $page = sanitize_text_field(wp_unslash($_GET['page'] ?? ''));

    if (empty($page) || $page !== 'elyssa-ai') {
        return;
    }

    wp_enqueue_script('elyssa-ai-js', plugin_dir_url(__FILE__) . '../editor/js/elyssa-ai.js?'.time(), array('jquery'), '1.0', true);
    wp_set_script_translations( 'elyssa-ai-js', 'elyssa-ai' );

    wp_localize_script('elyssa-ai-js', 'elyssaAiVars', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('elyssa_ai_nonce'),
    ));
}
add_action('admin_enqueue_scripts', 'elyssa_ai_enqueue_scripts');

add_action('wp_ajax_elyssa_ai_get_prompt_content', 'elyssa_ai_get_prompt_content_callback');

function elyssa_ai_get_prompt_content_callback() {
    check_ajax_referer('elyssa_ai_nonce', 'nonce');

    if (!current_user_can('edit_posts')) {
        wp_send_json_error(array('message' => __('Insufficient permissions', 'elyssa-ai')));
        return;
    }

    $prompt_id = isset($_POST['prompt_id']) ? intval($_POST['prompt_id']) : 0;

    if (!$prompt_id) {
        wp_send_json_error(array('message' => __('Invalid prompt ID', 'elyssa-ai')));
        return;
    }

    $user_prompt = get_post_meta($prompt_id, '_user_prompt', true);

    if ($user_prompt) {
        wp_send_json_success(array('user_prompt' => $user_prompt));
    } else {
        wp_send_json_error(array('message' => __('Prompt not found', 'elyssa-ai')));
    }
}

function elyssa_ai_add_generate_button() {
    global $pagenow;

    if ($pagenow != 'edit.php') {
        return;
    }
    
    // Registriere und enqueue das Script
    wp_register_script(
        'elyssa-ai-generate-button-js',
        '',
        array(),
        ELYSSA_AI_VERSION,
        true
    );
    
    wp_enqueue_script('elyssa-ai-generate-button-js');
    
    // Füge den inline Code hinzu
    $script = "document.addEventListener('DOMContentLoaded', function() {";
    $script .= "    var addNewButton = document.querySelector('.page-title-action');";
    $script .= "    if (addNewButton) {";
    $script .= "        var wrapperDiv = document.createElement('div');";
    $script .= "        wrapperDiv.style.display = 'inline-block';";
    $script .= "        wrapperDiv.style.verticalAlign = 'middle';";
    $script .= "        wrapperDiv.style.marginLeft = '10px';";
    $script .= "        wrapperDiv.style.marginTop = '-3px';";
    $script .= "        var generateButton = document.createElement('button');";
    $script .= "        generateButton.className = 'elyssa-ai-button';";
    $script .= "        generateButton.style.display = 'inline-flex';";
    $script .= "        generateButton.style.alignItems = 'center';";
    $script .= "        generateButton.innerHTML = '<img src=\"" . esc_url(plugins_url('elyssa-ai/images/elyssa-icon-white.webp')) . "\" alt=\"elyssa AI Logo\" style=\"width: 15px; height: 15px; margin-right: 5px;\"> " . esc_js(__('elyssa AI Generate', 'elyssa-ai')) . "';";
    $script .= "        generateButton.addEventListener('click', function() {";
    $script .= "            window.location.href = '" . esc_url(admin_url('admin.php?page=elyssa-ai')) . "';";
    $script .= "        });";
    $script .= "        wrapperDiv.appendChild(generateButton);";
    $script .= "        addNewButton.parentNode.insertBefore(wrapperDiv, addNewButton.nextSibling);";
    $script .= "    }";
    $script .= "});";
    
    wp_add_inline_script('elyssa-ai-generate-button-js', $script);
}
add_action('admin_print_scripts-edit.php', 'elyssa_ai_add_generate_button');

function elyssa_is_text_enabled() {
    return get_option('elyssa_ai_text_service') && get_option('elyssa_ai_text_model') && get_option('elyssa_ai_text_api_key');
}

function elyssa_is_image_enabled() {
    return get_option('elyssa_ai_image_service') && get_option('elyssa_ai_image_model') && get_option('elyssa_ai_image_api_key');
}

function attempt_json_fix($response) {
    // Entferne alles vor dem ersten { und nach dem letzten }
    $response = preg_replace('/^[^{]*/', '', $response);
    $response = preg_replace('/[^}]*$/', '', $response);

    // Ersetze einfache Anführungszeichen durch doppelte
    $response = str_replace("'", '"', $response);

    // Füge Anführungszeichen zu Schlüsseln hinzu, die keine haben
    $response = preg_replace('/(\w+)(?=\s*:)/', '"$1"', $response);

    return $response;
}

function parse_structured_content($content)
{
    $data = [
        'title' => '',
        'content' => '',
        'keywords' => []
    ];

    // Extrahiere den Titel
    if (preg_match('/TITLE:\s*(.+)/i', $content, $matches)) {
        $data['title'] = trim($matches[1]);
    }

    // Extrahiere den Content
    if (preg_match('/CONTENT:\s*(.+?)(?=KEYWORDS:|$)/is', $content, $matches)) {
        $data['content'] = trim($matches[1]);
        $data['content'] = str_replace('<h1>'.$data['title'].'</h1>', '', $data['content']);
        $data['content'] = preg_replace('/\s*KEYWORDS:.*$/is', '', $data['content']);
        $data['content'] = trim($data['content']);
    }

    // Extrahiere die Keywords
    if (preg_match('/KEYWORDS:\s*(.+)/i', $content, $matches)) {
        $data['keywords'] = array_map('trim', explode(',', $matches[1]));
    }

    return $data;
}

/**
 * Verschlüsselt einen API-Schlüssel mit dem WordPress Salting-System
 * 
 * @param string $api_key Der zu verschlüsselnde API-Schlüssel
 * @return string Der verschlüsselte API-Schlüssel
 */
function elyssa_ai_encrypt_api_key($api_key) {
    if (empty($api_key)) {
        return '';
    }
    
    // Wir nutzen den Authentication Unique Key aus wp-config.php als Basis für unseren Schlüssel
    $salt = defined('AUTH_KEY') ? AUTH_KEY : 'elyssa-ai-fallback-salt';
    
    // Sicherstellen, dass der Schlüssel stark genug ist
    $key = hash('sha256', $salt);
    
    // Erstellen eines zufälligen Initialization Vector (IV)
    $iv_size = openssl_cipher_iv_length('AES-256-CBC');
    $iv = openssl_random_pseudo_bytes($iv_size);
    
    // Verschlüssele den API-Schlüssel
    $encrypted = openssl_encrypt($api_key, 'AES-256-CBC', $key, 0, $iv);
    
    // Kombiniere IV und verschlüsselten Text für die Speicherung
    $encrypted_with_iv = base64_encode($iv . $encrypted);
    
    return $encrypted_with_iv;
}

/**
 * Entschlüsselt einen API-Schlüssel
 * 
 * @param string $encrypted_api_key Der verschlüsselte API-Schlüssel
 * @return string Der entschlüsselte API-Schlüssel
 */
function elyssa_ai_decrypt_api_key($encrypted_api_key) {
    if (empty($encrypted_api_key)) {
        return '';
    }
    
    try {
        // Den gespeicherten String dekodieren
        $decoded = base64_decode($encrypted_api_key);
        if ($decoded === false) {
            elyssa_ai_log("Base64 decoding failed for API key");
            return '';
        }
        
        // Wir nutzen den Authentication Unique Key aus wp-config.php als Basis für unseren Schlüssel
        $salt = defined('AUTH_KEY') ? AUTH_KEY : 'elyssa-ai-fallback-salt';
        $key = hash('sha256', $salt);
        
        // IV-Größe ermitteln und IV extrahieren
        $iv_size = openssl_cipher_iv_length('AES-256-CBC');
        if (strlen($decoded) <= $iv_size) {
            elyssa_ai_log("Encrypted data too short");
            return '';
        }
        
        $iv = substr($decoded, 0, $iv_size);
        $encrypted = substr($decoded, $iv_size);
        
        // Entschlüsseln
        $decrypted = openssl_decrypt($encrypted, 'AES-256-CBC', $key, 0, $iv);
        
        if ($decrypted === false) {
            elyssa_ai_log("Decryption failed for API key");
            return '';
        }
        
        return $decrypted;
    } catch (Exception $e) {
        elyssa_ai_log("Error decrypting API key: " . $e->getMessage());
        return '';
    }
}

/**
 * Hilfsfunktion zum Umgang mit API-Schlüsseln in der Datenbank
 * 
 * @param string $option_name Der Name der Option
 * @param string $value Der neue Wert (API-Schlüssel)
 * @return bool Ergebnis der Update-Operation
 */
function elyssa_ai_update_api_key_option($option_name, $value) {
    if (empty($value)) {
        return update_option($option_name, '');
    }
    
    $encrypted_value = elyssa_ai_encrypt_api_key($value);
    return update_option($option_name, $encrypted_value);
}

/**
 * Hilfsfunktion zum Abrufen von API-Schlüsseln aus der Datenbank
 * 
 * @param string $option_name Der Name der Option
 * @param string $default Der Standardwert, wenn keine Option gefunden wird
 * @return string Der entschlüsselte API-Schlüssel oder der Standardwert
 */
function elyssa_ai_get_api_key_option($option_name, $default = '') {
    $encrypted_value = get_option($option_name, '');
    
    if (empty($encrypted_value)) {
        return $default;
    }
    
    // Prüfen, ob der Wert bereits verschlüsselt ist
    // Wenn die Option noch im Klartext vorliegt, geben wir sie zurück und empfehlen eine einmalige Aktualisierung
    if (strpos($encrypted_value, 'sk-') === 0 || strpos($encrypted_value, 'bearer') === 0) {
        elyssa_ai_log("API key for {$option_name} is not encrypted. Consider updating the option.");
        return $encrypted_value;
    }
    
    return elyssa_ai_decrypt_api_key($encrypted_value);
}