# AI Services Implementation Guide

This directory contains the service implementations for different AI providers. Each service must follow the standardized structure and implement the abstract `AI_Service` class.

## Creating a New Service

### 1. Create Service File
Create a new file in the `includes/services` directory:
```php
// includes/services/class-yourservice-service.php
<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * YourService Implementation
 *
 * @package ElyssaAI
 * @since 2.0.0
 */
class YourService_Service extends AI_Service {
    // Implementation here
}
```

### 2. Implement Required Properties
```php
/** @var string API key */
private $api_key;

/** @var string Service type (text/image) */
private $service_type;

/** @var string Selected AI model */
private $model;
```

### 3. Implement Constructor
```php
public function __construct($type, $model = null) {
    $this->service_type = $type;
    $this->api_key = get_option('elyssa_ai_' . $type . '_api_key');
    $this->model = $model ?? 'default-model';
}
```

### 4. Define Service Information
```php
public static function get_service_info(): array {
    return [
        'id' => 'your-service',
        'name' => 'Your Service Name',
        'version' => '1.0.0',
        'description' => 'Service description',
        'types' => ['text' and/or 'image'],
        'features' => [
            // List features like 'system_messages', 'function_calling', etc.
        ],
        'limits' => [
            'max_tokens' => 4096,
            'max_context_length' => 32768,
            // Add other limits...
        ],
        'endpoints' => [
            'text_completion' => 'https://api.example.com/v1/completion',
            // Add other endpoints...
        ],
    ];
}
```

### 5. Implement Text Generation
```php
public function generate_text(string $prompt, array $options = [], ?string $system = null): string|WP_Error {
    if ($this->service_type !== 'text') {
        return new WP_Error(
            'service_error',
            esc_html__('Not configured for text generation', 'elyssa-ai')
        );
    }

    // Implement API call and response handling
    // Return string on success or WP_Error on failure
}
```

### 6. Implement Image Generation
```php
public function generate_image(string $prompt, array $options = []): string|WP_Error {
    if ($this->service_type !== 'image') {
        return new WP_Error(
            'service_error',
            esc_html__('Not configured for image generation', 'elyssa-ai')
        );
    }

    // Implement API call and response handling
    // Return image URL on success or WP_Error on failure
}
```

### 7. Implement Configuration Validation
```php
public function validate_config(): bool|WP_Error {
    if (empty($this->api_key)) {
        return new WP_Error(
            'service_error',
            esc_html__('API key not configured', 'elyssa-ai')
        );
    }

    $models = self::get_service_info()['models'][$this->service_type];
    if (!isset($models[$this->model])) {
        return new WP_Error(
            'service_error',
            esc_html__('Invalid model selected', 'elyssa-ai')
        );
    }

    return true;
}
```

## Best Practices

1. **Error Handling**
   - Use service-specific error codes (e.g., 'openai_error', 'anthropic_error')
   - Always return WP_Error for failures
   - Include meaningful error messages

2. **API Integration**
   - Use WordPress HTTP functions (wp_remote_post, wp_remote_get)
   - Set appropriate timeouts
   - Handle rate limits
   - Log API requests with elyssa_ai_log()

3. **Security**
   - Validate all inputs
   - Sanitize outputs
   - Use WordPress nonces where appropriate
   - Never expose API keys

4. **Documentation**
   - Include complete PHPDoc blocks
   - Document all public methods
   - Include usage examples
   - Document API endpoints

## Testing

Before submitting a new service:

1. Test all implemented methods
2. Verify error handling
3. Check API response handling
4. Validate configuration checks
5. Test with different model options

## Example Services

See existing implementations for reference:
- OpenAI_Service (Text + Image)
- Anthropic_Service (Text)
- Leonardo_Service (Image)
- Flux_Service (Image)