<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * OpenAI Service Implementation
 *
 * Handles both text and image generation using the OpenAI API.
 *
 * @package ElyssaAI
 * @since 2.0.0
 */
class ElyssaAI_OpenAIService extends ElyssaAI_Service
{
    /** @var string API key for OpenAI */
    private $api_key;

    /** @var string Service type (text/image) */
    private $service_type;

    /** @var string Selected AI model */
    private $model;

    /**
     * Initialize the OpenAI service
     *
     * @param string $type Service type (text/image)
     * @param string|null $model Selected model ID
     */
    public function __construct($type, $model = null)
    {
        $this->service_type = $type;
        // API-Schlüssel entschlüsselt abrufen
        $this->api_key = elyssa_ai_get_api_key_option('elyssa_ai_' . $type . '_api_key');
        $this->model = $model ?? ($type === 'text' ? 'gpt-4' : 'dall-e-3');
    }

    /**
     * Get service information and capabilities
     *
     * @return array Service information
     */
    public static function get_service_info(): array
    {
        return [
            'id' => 'openai',
            'name' => 'OpenAI',
            'version' => '1.0.0',
            'description' => 'Advanced AI models for text and image generation',
            'types' => ['text', 'image'],
            'features' => [
                'streaming',
                'function_calling',
                'system_messages',
                'image_variations',
            ],
            'limits' => [
                'max_tokens' => 4096,
                'max_image_size' => 1024,
                'max_context_length' => 8192,
            ],
            'endpoints' => [
                'text_completion' => 'https://api.openai.com/v1/chat/completions',
                'image_generation' => 'https://api.openai.com/v1/images/generations',
            ],
        ];
    }

    /**
     * Generate text using OpenAI
     *
     * @param string $prompt The input prompt for text generation
     * @param array $options Additional options for generation
     * @param string|null $system System message for the AI
     * @return string|WP_Error Generated text or error
     */
    public function generate_text(string $prompt, array $options = [], ?string $system = null)
    {
        if ($this->service_type !== 'text') {
            return new WP_Error(
                'openai_error',
                esc_html__('This service instance is not configured for text generation', 'elyssa-ai')
            );
        }

        $service_info = self::get_service_info();
        $api_url = $service_info['endpoints']['text_completion'];
        
        $messages = [];
        if ($system !== null) {
            $messages[] = ['role' => 'system', 'content' => $system];
        }
        $messages[] = ['role' => 'user', 'content' => $prompt];

        $data = [
            'model' => $this->model,
            'messages' => $messages,
            'temperature' => $options['temperature'] ?? 0.7,
            'max_tokens' => $options['max_tokens'] ?? 1000
        ];

        elyssa_ai_log($data);

        $response = wp_remote_post($api_url, [
            'headers' => [
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'application/json'
            ],
            'body' => wp_json_encode($data),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($body['choices'][0]['message']['content'])) {
            return $body['choices'][0]['message']['content'];
        }

        return new WP_Error(
            'openai_error',
            $body['error']['message'] ?? esc_html__('Unexpected response format from OpenAI API', 'elyssa-ai')
        );
    }

    /**
     * Generate image using OpenAI
     *
     * @param string $prompt The input prompt for image generation
     * @param array $options Additional options for generation
     * @return string|WP_Error URL of generated image or error
     */
    public function generate_image(string $prompt, array $options = [])
    {
        if ($this->service_type !== 'image') {
            return new WP_Error(
                'openai_error',
                esc_html__('This service instance is not configured for image generation', 'elyssa-ai')
            );
        }

        $service_info = self::get_service_info();
        $api_url = $service_info['endpoints']['image_generation'];

        $data = [
            'model' => $this->model,
            'prompt' => $prompt,
            'n' => $options['n'] ?? 1,
            'size' => $options['size'] ?? '1024x1024',
            'response_format' => 'url',
        ];

        elyssa_ai_log($data);

        $response = wp_remote_post($api_url, [
            'headers' => [
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'application/json'
            ],
            'body' => wp_json_encode($data),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($body['data'][0]['url'])) {
            return $body['data'][0]['url'];
        }

        return new WP_Error(
            'openai_error',
            $body['error']['message'] ?? esc_html__('Unexpected response format from OpenAI API', 'elyssa-ai')
        );
    }

    /**
     * Validate service configuration
     *
     * @return bool|WP_Error True if valid, WP_Error if invalid
     */
    public function validate_config()
    {
        if (empty($this->api_key)) {
            return new WP_Error(
                'openai_error',
                esc_html__('OpenAI API key is not configured', 'elyssa-ai')
            );
        }

        if (empty($this->model)) {
            return new WP_Error(
                'openai_error',
                esc_html__('No model specified for OpenAI', 'elyssa-ai')
            );
        }

        return true;
    }
}
