<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * StabilityAI Service Implementation
 *
 * Handles image generation using the Stability AI API.
 *
 * @package ElyssaAI
 * @since 2.0.0
 */
class StabilityAI_Service extends ElyssaAI_Service
{
    /** @var string API key for StabilityAI */
    private $api_key;

    /** @var string Service type (text/image) */
    private $service_type;

    /** @var string Selected AI model */
    private $model;

    /**
     * Initialize the StabilityAI service
     *
     * @param string $type  Service type (text/image)
     * @param string|null $model Selected model ID
     */
    public function __construct($type, $model = null)
    {
        $this->service_type = $type;
        $this->model = $model ?? 'sd3-medium';
        $this->api_key = get_option('elyssa_ai_image_api_key');
    }

    /**
     * Get service information and capabilities
     *
     * @return array Service information
     */
    public static function get_service_info(): array
    {
        return [
            'id' => 'stabilityai',
            'name' => 'Stability AI',
            'version' => '1.0.0',
            'description' => 'Advanced image generation using Stable Diffusion models',
            'types' => ['image'],
            'features' => [
                'negative_prompts',
                'seed_control',
                'size_control',
            ],
            'limits' => [
                'max_image_size' => 1024,
                'min_image_size' => 512,
                'max_samples' => 4,
            ],
            'endpoints' => [
                'image_generation' => 'https://api.stability.ai/v1/generation/{model}/text-to-image',
            ],
        ];
    }

    /**
     * Generate text using the service
     * 
     * @throws Exception This service does not support text generation
     */
    public function generate_text(string $prompt, array $options = [], ?string $system = null): string
    {
        throw new Exception(esc_html__('This service does not support text generation', 'elyssa-ai'));
    }

    /**
     * Generate image using StabilityAI
     *
     * @param string $prompt  The image generation prompt
     * @param array  $options Generation options
     * @return string|WP_Error URL of generated image or error
     */
    public function generate_image(string $prompt, array $options = []): string|WP_Error
    {
        if ($this->service_type !== 'image') {
            return new WP_Error(
                'stabilityai_error',
                esc_html__('This service instance is not configured for image generation', 'elyssa-ai')
            );
        }

        $service_info = self::get_service_info();
        $api_url = str_replace('{model}', $this->model, $service_info['endpoints']['image_generation']);

        $request = [
            "text_prompts" => [
                [
                    "text" => $prompt
                ]
            ],
            "cfg_scale" => 7,
            "height" => $options['height'] ?? 1024,
            "width" => $options['width'] ?? 1024,
            "samples" => $options['samples'] ?? 1,
            "steps" => $options['steps'] ?? 30,
            "output_format" => "jpeg"
        ];

        if (isset($options['seed'])) {
            $request['seed'] = intval($options['seed']);
        }

        if (isset($options['negative_prompt'])) {
            $request['negative_prompt'] = $options['negative_prompt'];
        }

        elyssa_ai_log($request);

        $response = wp_remote_post($api_url, [
            'timeout' => 30,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'Authorization' => "Bearer {$this->api_key}",
            ],
            'body' => json_encode($request)
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $http_status = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        elyssa_ai_log($body);

        if ($http_status !== 200) {
            return new WP_Error('stabilityai_error', $body);
        }

        $body = json_decode($body, true);

        if (isset($body['artifacts'][0]['base64'])) {
            return $this->save_generated_image($body['artifacts'][0]['base64']);
        }

        return new WP_Error(
            'stabilityai_error', 
            $body['message'] ?? esc_html__('Unexpected response format from Stability AI API', 'elyssa-ai')
        );
    }

    /**
     * Validate service configuration
     *
     * @return bool|WP_Error True if valid, WP_Error if invalid
     */
    public function validate_config(): bool|WP_Error
    {
        if (empty($this->api_key)) {
            return new WP_Error(
                'stabilityai_error',
                esc_html__('StabilityAI API key is not configured', 'elyssa-ai')
            );
        }

        if (!in_array($this->model, array_keys(self::get_service_info()['models']['image']))) {
            return new WP_Error(
                'stabilityai_error',
                esc_html__('Invalid model selected for StabilityAI', 'elyssa-ai')
            );
        }

        return true;
    }

    /**
     * Save the generated base64 image
     *
     * @param string $base64_image Base64 encoded image data
     * @return string|WP_Error URL of saved image or error
     */
    private function save_generated_image(string $base64_image): string|WP_Error
    {
        $upload_dir = wp_upload_dir();
        $filename = 'stability_ai_' . time() . '.jpg';
        $file_path = $upload_dir['path'] . '/' . $filename;
        $file_url = $upload_dir['url'] . '/' . $filename;

        $image_data = base64_decode($base64_image);
        if ($image_data === false) {
            return new WP_Error('stabilityai_error', esc_html__('Failed to decode base64 image data', 'elyssa-ai'));
        }

        $save_result = file_put_contents($file_path, $image_data);
        if ($save_result === false) {
            return new WP_Error('stabilityai_error', esc_html__('Failed to save image file', 'elyssa-ai'));
        }

        return $file_url;
    }
}