<?php
if (!defined('ABSPATH')) {
    exit;
}
class ElyssaAIPrompts extends ElyssaAIBaseService
{
    private $capability = 'edit_posts';
    private $menu_slug = 'edit.php?post_type=elyssa_ai_prompt';

    public function __construct() {
        add_action('init', array($this, 'register_prompt_post_type'));
        add_action('init', array($this, 'register_prompt_taxonomy'));
        add_action('admin_menu', array($this, 'adjust_admin_menu'), 999);
        add_action('add_meta_boxes', array($this, 'add_prompt_meta_boxes'));
        add_action('save_post_elyssa_ai_prompt', array($this, 'save_prompt_meta'));
        add_filter('manage_elyssa_ai_prompt_posts_columns', array($this, 'prompt_columns'));
        add_action('manage_elyssa_ai_prompt_posts_custom_column', array($this, 'prompt_column_content'), 10, 2);

        add_filter('posts_join', array($this, 'search_join'));
        add_filter('posts_where', array($this, 'search_where'));
        add_filter('posts_groupby', array($this, 'search_groupby'));

        add_filter('parent_file', array($this, 'fix_admin_parent_file'));
        add_filter('submenu_file', array($this, 'fix_admin_submenu_file'));

        add_action('add_meta_boxes', array($this, 'add_preview_meta_box'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_preview_scripts'));
        add_action('wp_ajax_elyssa_ai_preview_prompt', array($this, 'ajax_preview_prompt'));

        add_action('restrict_manage_posts', array($this, 'add_type_filter'));
        add_action('restrict_manage_posts', array($this, 'add_category_filter'));
        add_filter('parse_query', array($this, 'filter_prompts_by_type'));
        add_filter('parse_query', array($this, 'filter_prompts_by_category'));

        add_action('rest_api_init', array($this, 'register_rest_fields'));
    }

    public function register_prompt_post_type() {
        $labels = array(
            'name'               => esc_html_x('Prompt Database', 'post type general name', 'elyssa-ai'),
            'singular_name'      => esc_html_x('Prompt', 'post type singular name', 'elyssa-ai'),
            'menu_name'          => esc_html_x('Prompts', 'admin menu', 'elyssa-ai'),
            'name_admin_bar'     => esc_html_x('Prompt', 'add new on admin bar', 'elyssa-ai'),
            'add_new'            => esc_html_x('Add New', 'prompt', 'elyssa-ai'),
            'add_new_item'       => esc_html__('Add New Prompt', 'elyssa-ai'),
            'new_item'           => esc_html__('New Prompt', 'elyssa-ai'),
            'edit_item'          => esc_html__('Edit Prompt', 'elyssa-ai'),
            'view_item'          => esc_html__('View Prompt', 'elyssa-ai'),
            'all_items'          => esc_html__('Prompt DB', 'elyssa-ai'),
            'search_items'       => esc_html__('Search Prompts', 'elyssa-ai'),
            'parent_item_colon'  => esc_html__('Parent Prompts:', 'elyssa-ai'),
            'not_found'          => esc_html__('No prompts found.', 'elyssa-ai'),
            'not_found_in_trash' => esc_html__('No prompts found in Trash.', 'elyssa-ai')
        );

        $args = array(
            'labels'             => $labels,
            'public'             => false,
            'publicly_queryable' => false,
            'show_ui'            => true,
            'show_in_menu'       => 'elyssa-ai',
            'query_var'          => true,
            'rewrite'            => array('slug' => 'prompt'),
            'capability_type'    => 'post',
            'capabilities'       => array(
                'edit_post'          => $this->capability,
                'read_post'          => $this->capability,
                'delete_post'        => $this->capability,
                'edit_posts'         => $this->capability,
                'edit_others_posts'  => $this->capability,
                'publish_posts'      => $this->capability,
                'read_private_posts' => $this->capability
            ),
            'has_archive'        => false,
            'hierarchical'       => false,
            'menu_position'      => null,
            'supports'           => array('title'),
            'taxonomies'         => array('elyssa_ai_prompt_category')
        );

        register_post_type('elyssa_ai_prompt', $args);
    }

    public function register_prompt_taxonomy() {
        $labels = array(
            'name'              => _x('Prompt Categories', 'taxonomy general name', 'elyssa-ai'),
            'singular_name'     => _x('Prompt Category', 'taxonomy singular name', 'elyssa-ai'),
            'search_items'      => __('Search Prompt Categories', 'elyssa-ai'),
            'all_items'         => __('All Prompt Categories', 'elyssa-ai'),
            'parent_item'       => __('Parent Prompt Category', 'elyssa-ai'),
            'parent_item_colon' => __('Parent Prompt Category:', 'elyssa-ai'),
            'edit_item'         => __('Edit Prompt Category', 'elyssa-ai'),
            'update_item'       => __('Update Prompt Category', 'elyssa-ai'),
            'add_new_item'      => __('Add New Prompt Category', 'elyssa-ai'),
            'new_item_name'     => __('New Prompt Category Name', 'elyssa-ai'),
            'menu_name'         => __('Prompt Categories', 'elyssa-ai'),
        );

        $args = array(
            'hierarchical'      => true,
            'labels'            => $labels,
            'show_ui'           => true,
            'show_admin_column' => true,
            'query_var'         => true,
            'rewrite'           => array('slug' => 'prompt-category'),
        );

        register_taxonomy('elyssa_ai_prompt_category', array('elyssa_ai_prompt'), $args);
    }

    public function register_rest_fields() {
        register_rest_route('wp/v2', '/elyssa_ai_prompt/(?P<id>\d+)', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_prompt'),
            'permission_callback' => function() {
                return current_user_can('edit_posts');
            }
        ));
    }

    public function adjust_admin_menu() {
        global $submenu;

        if (isset($submenu['elyssa-ai'])) {
            $prompt_menu_item = null;

            foreach ($submenu['elyssa-ai'] as $key => $item) {
                if ($item[2] === 'edit.php?post_type=elyssa_ai_prompt') {
                    $prompt_menu_item = $submenu['elyssa-ai'][$key];
                    unset($submenu['elyssa-ai'][$key]);
                    break;
                }
            }

            foreach ($submenu['elyssa-ai'] as $key => $item) {
                if ($item[2] === 'post-new.php?post_type=elyssa_ai_prompt') {
                    unset($submenu['elyssa-ai'][$key]);
                    break;
                }
            }

            if ($prompt_menu_item) {
                $prompt_menu_item[0] = esc_html__('Prompt DB', 'elyssa-ai');
                $submenu['elyssa-ai'][] = $prompt_menu_item;
            }
        }
    }

    public function add_prompt_meta_boxes() {
        if (!current_user_can($this->capability)) {
            return;
        }
        add_meta_box(
            'elyssa_ai_prompt_fields',
            esc_html__('Prompt Details', 'elyssa-ai'),
            array($this, 'prompt_meta_box_callback'),
            'elyssa_ai_prompt',
            'normal',
            'high'
        );
        add_meta_box(
            'elyssa_ai_prompt_preview',
            esc_html__('Prompt Preview', 'elyssa-ai'),
            array($this, 'render_preview_meta_box'),
            'elyssa_ai_prompt',
            'normal',
            'high'
        );
    }

    public function add_type_filter() {
        global $typenow;
        if ($typenow == 'elyssa_ai_prompt') {
            $current = sanitize_text_field(wp_unslash($_GET['prompt_type'] ?? ''));
            ?>
            <select name="prompt_type">
                <option value=""><?php esc_html_e('All Types', 'elyssa-ai'); ?></option>
                <option value="text" <?php selected($current, 'text'); ?>><?php esc_html_e('Text', 'elyssa-ai'); ?></option>
                <option value="image" <?php selected($current, 'image'); ?>><?php esc_html_e('Image', 'elyssa-ai'); ?></option>
            </select>
            <?php
        }
    }

    public function filter_prompts_by_type($query) {
        global $pagenow;
        $type = sanitize_text_field(wp_unslash($_GET['prompt_type'] ?? ''));
        if (is_admin() && $pagenow=='edit.php' && isset($_GET['post_type']) && $_GET['post_type'] == 'elyssa_ai_prompt' && $type != '') {
            $query->query_vars['meta_key'] = '_prompt_type';
            $query->query_vars['meta_value'] = $type;
        }
    }

    public static function get_prompts_for_dropdown($type = 'text') {
        $args = array(
            'post_type' => 'elyssa_ai_prompt',
            'posts_per_page' => -1,
            'orderby' => 'title',
            'order' => 'ASC',
            'meta_query' => array(
                array(
                    'key' => '_prompt_type',
                    'value' => $type,
                    'compare' => '='
                )
            )
        );
        $prompts = get_posts($args);

        $prompt_options = array();
        foreach ($prompts as $prompt) {
            $prompt_options[$prompt->ID] = $prompt->post_title;
        }

        return $prompt_options;
    }

    public function get_prompt($request) {
        $prompt_id = $request['id'];
        $prompt = get_post($prompt_id);

        if (!$prompt || $prompt->post_type !== 'elyssa_ai_prompt') {
            return new WP_Error('prompt_not_found', 'Prompt not found', array('status' => 404));
        }

        $response = array(
            'id' => $prompt->ID,
            'title' => $prompt->post_title,
            'meta' => array(
                '_user_prompt' => get_post_meta($prompt->ID, '_user_prompt', true)
            )
        );

        return new WP_REST_Response($response, 200);
    }

    public function prompt_meta_box_callback($post) {
        if (!current_user_can($this->capability)) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'elyssa-ai'));
        }

        wp_nonce_field('elyssa_ai_save_prompt_meta', 'elyssa_ai_prompt_meta_nonce');

        $user_prompt = get_post_meta($post->ID, '_user_prompt', true);
        $notes = get_post_meta($post->ID, '_notes', true);
        $type = get_post_meta($post->ID, '_prompt_type', true);
        ?>
        <p>
            <label for="elyssa_ai_prompt_type"><?php esc_html_e('Prompt Type:', 'elyssa-ai'); ?></label><br>
            <select id="elyssa_ai_prompt_type" name="elyssa_ai_prompt_type">
                <option value="text" <?php selected($type, 'text'); ?>><?php esc_html_e('Text', 'elyssa-ai'); ?></option>
                <option value="image" <?php selected($type, 'image'); ?>><?php esc_html_e('Image', 'elyssa-ai'); ?></option>
            </select>
        </p>
        <p>
            <label for="elyssa_ai_user_prompt"><?php esc_html_e('User Prompt:', 'elyssa-ai'); ?></label><br>
            <textarea id="elyssa_ai_user_prompt" name="elyssa_ai_user_prompt" rows="4" style="width:100%;"><?php echo esc_textarea($user_prompt); ?></textarea>
        </p>
        <p>
            <label for="elyssa_ai_notes"><?php esc_html_e('Notes:', 'elyssa-ai'); ?></label><br>
            <textarea id="elyssa_ai_notes" name="elyssa_ai_notes" rows="3" style="width:100%;"><?php echo esc_textarea($notes); ?></textarea>
        </p>
        <?php
    }

    public function save_prompt_meta($post_id) {
        if (!current_user_can($this->capability)) {
            return;
        }

        $nonce = sanitize_text_field(wp_unslash($_POST['elyssa_ai_prompt_meta_nonce'] ?? ''));
        if (empty($nonce) ||
            !wp_verify_nonce($nonce, 'elyssa_ai_save_prompt_meta')) {
            return;
        }

        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        if (!empty(sanitize_text_field(wp_unslash($_POST['elyssa_ai_prompt_type'])))) {
            update_post_meta($post_id, '_prompt_type', sanitize_text_field(wp_unslash($_POST['elyssa_ai_prompt_type'])));
        }

        if (!empty(sanitize_text_field(wp_unslash($_POST['elyssa_ai_user_prompt'])))) {
            update_post_meta($post_id, '_user_prompt', sanitize_textarea_field(wp_unslash($_POST['elyssa_ai_user_prompt'])));
        }

        if (!empty(sanitize_text_field(wp_unslash($_POST['elyssa_ai_notes'])))) {
            update_post_meta($post_id, '_notes', sanitize_textarea_field(wp_unslash($_POST['elyssa_ai_notes'])));
        }
    }

    public function prompt_columns($columns) {
        if (!current_user_can($this->capability)) {
            return $columns;
        }

        $columns = array(
            'cb' => $columns['cb'],
            'type' => __('Type', 'elyssa-ai'),
            'title' => __('Title', 'elyssa-ai'),
            'category' => __('Category', 'elyssa-ai'),
            'user_prompt' => __('User Prompt', 'elyssa-ai'),
            'notes' => __('Notes', 'elyssa-ai'),
            'date' => __('Date', 'elyssa-ai')
        );
        return $columns;
    }

    public function prompt_column_content($column, $post_id) {
        if (!current_user_can($this->capability)) {
            return;
        }

        switch ($column) {
            case 'type':
                $type = get_post_meta($post_id, '_prompt_type', true);
                echo esc_html(ucfirst($type));
                break;
            case 'category':
                $terms = get_the_terms($post_id, 'elyssa_ai_prompt_category');
                if (!empty($terms)) {
                    $term_names = array();
                    foreach ($terms as $term) {
                        $term_names[] = $term->name;
                    }
                    echo esc_html(implode(', ', $term_names));
                }
                break;
            case 'user_prompt':
                echo esc_html(wp_trim_words(get_post_meta($post_id, '_user_prompt', true), 10));
                break;
            case 'notes':
                echo esc_html(wp_trim_words(get_post_meta($post_id, '_notes', true), 10));
                break;
        }
    }

    public function search_join($join) {
        global $wpdb, $wp_query;

        if (!current_user_can($this->capability)) {
            return $join;
        }

        if (is_search() && !empty($wp_query->query_vars['s']) && $wp_query->query_vars['post_type'] == 'elyssa_ai_prompt') {
            $join .= " LEFT JOIN $wpdb->postmeta ON $wpdb->posts.ID = $wpdb->postmeta.post_id ";
        }
        return $join;
    }

    public function search_where($where) {
        global $wpdb, $wp_query;

        if (!current_user_can($this->capability)) {
            return $where;
        }

        if (is_search() && !empty($wp_query->query_vars['s']) && $wp_query->query_vars['post_type'] == 'elyssa_ai_prompt') {
            $search_term = $wp_query->query_vars['s'];
            $search_term = $wpdb->esc_like($search_term);
            $search_term = '%' . $search_term . '%';

            $where = preg_replace(
                "/\(\s*$wpdb->posts.post_title\s+LIKE\s*(\'[^\']+\')\s*\)/",
                "($wpdb->posts.post_title LIKE $1) 
                OR ($wpdb->postmeta.meta_key = '_system_prompt' AND $wpdb->postmeta.meta_value LIKE '$search_term') 
                OR ($wpdb->postmeta.meta_key = '_user_prompt' AND $wpdb->postmeta.meta_value LIKE '$search_term')
                OR ($wpdb->postmeta.meta_key = '_notes' AND $wpdb->postmeta.meta_value LIKE '$search_term')",
                $where
            );
        }
        return $where;
    }

    public function add_category_filter() {
        global $typenow;
        if ($typenow == 'elyssa_ai_prompt') {
            $taxonomy = 'elyssa_ai_prompt_category';
            $selected = sanitize_text_field(wp_unslash($_GET[$taxonomy] ?? ''));
            wp_dropdown_categories(array(
                'show_option_all' => __('All Categories', 'elyssa-ai'),
                'taxonomy'        => $taxonomy,
                'name'            => $taxonomy,
                'orderby'         => 'name',
                'selected'        => $selected,
                'hierarchical'    => true,
                'depth'           => 3,
                'show_count'      => true,
                'hide_empty'      => true,
            ));
        }
    }

    public function filter_prompts_by_category($query) {
        global $pagenow;
        $taxonomy = 'elyssa_ai_prompt_category';
        $q_vars = &$query->query_vars;
        if ($pagenow == 'edit.php' && isset($q_vars['post_type']) && $q_vars['post_type'] == 'elyssa_ai_prompt' && isset($q_vars[$taxonomy]) && is_numeric($q_vars[$taxonomy]) && $q_vars[$taxonomy] != 0) {
            $term = get_term_by('id', $q_vars[$taxonomy], $taxonomy);
            $q_vars[$taxonomy] = $term->slug;
        }
    }

    public function search_groupby($groupby) {
        global $wpdb, $wp_query;

        if (!current_user_can($this->capability)) {
            return $groupby;
        }

        if (is_search() && !empty($wp_query->query_vars['s']) && $wp_query->query_vars['post_type'] == 'elyssa_ai_prompt') {
            $groupby = "$wpdb->posts.ID";
        }
        return $groupby;
    }

    public function fix_admin_parent_file($parent_file) {
        global $current_screen, $pagenow;

        if ($current_screen->post_type == 'elyssa_ai_prompt') {
            $parent_file = 'elyssa-ai';
        }

        return $parent_file;
    }

    public function fix_admin_submenu_file($submenu_file) {
        global $current_screen, $pagenow;

        if ($current_screen->post_type == 'elyssa_ai_prompt') {
            $submenu_file = $this->menu_slug;
        }

        return $submenu_file;
    }

    public function add_preview_meta_box() {
        add_meta_box(
            'elyssa_ai_prompt_preview',
            esc_html__('Prompt Preview', 'elyssa-ai'),
            array($this, 'render_preview_meta_box'),
            'elyssa_ai_prompt',
            'normal',
            'high'
        );
    }

    public function render_preview_meta_box($post) {
        wp_nonce_field('elyssa_ai_preview_prompt', 'elyssa_ai_preview_prompt_nonce');

        $cached_preview = get_post_meta($post->ID, '_elyssa_ai_preview_cache', true);
        $prompt_type = get_post_meta($post->ID, '_prompt_type', true);
        ?>
        <div id="elyssa-ai-preview-container">
            <button id="elyssa-ai-generate-preview" class="elyssa-ai-button">
                <?php esc_html_e('Generate Preview', 'elyssa-ai'); ?>
            </button>
            <div id="elyssa-ai-preview-result" style="margin-top: 10px; <?php echo $cached_preview ? '' : 'display: none;'; ?>">
                <h4><?php esc_html_e('Preview Result:', 'elyssa-ai'); ?></h4>
                <div id="elyssa-ai-preview-content"></div>
            </div>
        </div>
        <?php
    }

    public function enqueue_preview_scripts($hook) {
        global $post;

        if ($hook == 'post-new.php' || $hook == 'post.php') {
            if ('elyssa_ai_prompt' === $post->post_type) {
                wp_enqueue_script('elyssa-ai-preview', plugin_dir_url(__FILE__) . 'assets/elyssa-ai-preview.js?'.time(), array('jquery'), '1.0', true);
                
                // Holen der benötigten Daten, die sonst im direkten Script waren
                $cached_preview = get_post_meta($post->ID, '_elyssa_ai_preview_cache', true);
                $prompt_type = get_post_meta($post->ID, '_prompt_type', true);
                
                wp_localize_script('elyssa-ai-preview', 'elyssaAiPreview', array(
                    'ajax_url' => admin_url('admin-ajax.php'),
                    'nonce' => wp_create_nonce('elyssa_ai_preview_prompt'),
                    'generating' => esc_html__('Generating...', 'elyssa-ai'),
                    'generate_preview' => esc_html__('Generate Preview', 'elyssa-ai'),
                    'error_occurred' => esc_html__('An error occurred while generating the preview.', 'elyssa-ai'),
                    'cachedPreview' => $cached_preview,
                    'promptType' => $prompt_type
                ));
            }
        }
    }

    public function ajax_preview_prompt() {
        check_ajax_referer('elyssa_ai_preview_prompt', 'nonce');

        if (!current_user_can($this->capability)) {
            wp_send_json_error(array('message' => esc_html__('Insufficient permissions', 'elyssa-ai')));
            return;
        }

        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $user_prompt = sanitize_textarea_field(wp_unslash($_POST['user_prompt'] ?? ''));
        $prompt_type = sanitize_text_field(wp_unslash($_POST['prompt_type'] ?? 'text'));

        // Check if we have a cached preview
        $cached_preview = get_post_meta($post_id, '_elyssa_ai_preview_cache', true);
        $cached_user_prompt = get_post_meta($post_id, '_elyssa_ai_user_prompt_cache', true);

        if ($cached_preview && $cached_user_prompt === $user_prompt) {
            wp_send_json_success($cached_preview);
            return;
        }

        $preview_content = $this->generate_preview($user_prompt, $prompt_type);

        if (is_wp_error($preview_content)) {
            wp_send_json_error(array('message' => $preview_content->get_error_message()));
        } else {
            if ($prompt_type === 'text') {
                $parsed_content = $this->parse_structured_content($preview_content);
                $preview_content = $parsed_content;
            } elseif ($prompt_type === 'image') {
                $preview_content = $this->download_image_to_media($preview_content, $user_prompt);
            }

            // Cache the new preview
            update_post_meta($post_id, '_elyssa_ai_preview_cache', $preview_content);
            update_post_meta($post_id, '_elyssa_ai_user_prompt_cache', $user_prompt);

            wp_send_json_success($preview_content);
        }
    }

    private function download_image_to_media($image_url, $prompt) {
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');

        $tmp = download_url($image_url);
        if (is_wp_error($tmp)) {
            return $tmp;
        }

        $file_array = array(
            'name' => 'elyssa_ai_preview_' . uniqid() . '.png',
            'tmp_name' => $tmp
        );

        $id = media_handle_sideload($file_array, 0);

        if (is_wp_error($id)) {
            @wp_delete_file($file_array['tmp_name']);
            return $id;
        }

        $attachment_url = wp_get_attachment_url($id);

        // Set the image title and caption
        $attachment = array(
            'ID' => $id,
            'post_title' => 'AI Generated Image: ' . substr($prompt, 0, 50) . '...',
            'post_excerpt' => $prompt
        );
        wp_update_post($attachment);

        return array(
            'id' => $id,
            'url' => $attachment_url
        );
    }

    private function generate_preview($user_prompt, $prompt_type) {
        if ($prompt_type === 'text') {
            $init_result = $this->init_text_service();
            if (is_wp_error($init_result)) {
                return $init_result;
            }

            if (!$this->text_service) {
                return new WP_Error('ai_service_error', esc_html__('Text AI service is not available.', 'elyssa-ai'));
            }

            try {
                $prompt = sprintf(ServicePrompts::get_user('prompt_preview'), $user_prompt);
                $preview = $this->text_service->generate_text($prompt, ['max_tokens' => 150]);
                return $preview;
            } catch (Exception $e) {
                return new WP_Error('preview_error', $e->getMessage());
            }
        } elseif ($prompt_type === 'image') {
            $init_result = $this->init_image_service();
            if (is_wp_error($init_result)) {
                return $init_result;
            }

            if (!$this->image_service) {
                return new WP_Error('ai_service_error', esc_html__('Image AI service is not available.', 'elyssa-ai'));
            }

            try {
                $image_url = $this->image_service->generate_image($user_prompt);
                return $image_url;
            } catch (Exception $e) {
                return new WP_Error('preview_error', $e->getMessage());
            }
        } else {
            return new WP_Error('invalid_prompt_type', esc_html__('Invalid prompt type.', 'elyssa-ai'));
        }
    }

}

// Initialize the class
new ElyssaAIPrompts();