(function() {
    function addRateButton() {
        const bulkActionsElement = document.querySelector('.tablenav.top .bulkactions');
        if (bulkActionsElement) {
            const button = document.createElement('button');
            button.type = 'button';
            button.id = 'elyssa-rate-comments';
            button.className = 'elyssa-ai-button';

            if(!elyssaAiComments.ratingEnabled) {
                button.disabled = true;
                button.title = elyssaAiComments.ratingDisabledText;
            }

            const img = document.createElement('img');
            img.src = elyssaAiComments.pluginUrl + '/images/elyssa-icon-white.webp';
            img.alt = 'elyssa AI Logo';
            img.className = 'mr-2';
            img.style.width = '15px';
            img.style.height = '15px';
            img.style.marginRight = '5px';
            img.style.float = 'left';

            button.appendChild(img);
            button.appendChild(document.createTextNode(elyssaAiComments.rateButtonText));

            bulkActionsElement.parentNode.insertBefore(button, bulkActionsElement.nextSibling);

            initRateButton();
        }
    }

    function initRateButton() {
        const rateButton = document.getElementById('elyssa-rate-comments');
        if (rateButton) {
            rateButton.addEventListener('click', handleRateButtonClick);
        }
    }

    function handleRateButtonClick() {
        const unratedComments = Array.from(document.querySelectorAll('.elyssa-ai-rating-waiting'))
            .map(el => el.closest('tr').id.replace('comment-', ''));

        if (unratedComments.length === 0) {
            showAdminNotice(elyssaAiComments.noCommentsToRate, 'error');
            return;
        }

        showLoadingIndicator();
        this.disabled = true;
        this.textContent = elyssaAiComments.ratingInProgress;

        const totalComments = unratedComments.length;
        let completedComments = 0;

        Promise.all(unratedComments.map(commentId => rateComment(commentId, () => {
            completedComments++;
            updateProgress(completedComments, totalComments);
        })))
            .then(results => {
                const successfulRatings = results.filter(result => result.success).length;

                if (successfulRatings === totalComments) {
                    setAdminNotice(elyssaAiComments.allCommentsRated, 'success');
                } else {
                    setAdminNotice(
                        elyssaAiComments.someCommentsRated
                            .replace('{success}', successfulRatings)
                            .replace('{total}', totalComments),
                        'warning'
                    );
                }
            })
            .catch(() => {
                setAdminNotice(elyssaAiComments.ratingError, 'error');
            })
            .finally(() => {
                hideLoadingIndicator();
                this.disabled = false;
                this.textContent = elyssaAiComments.rateButtonText;
                location.reload();
            });
    }


    function rateComment(commentId, onComplete) {
        const formData = new FormData();
        formData.append('action', 'elyssa_rate_comments');
        formData.append('nonce', elyssaAiComments.nonce);
        formData.append('comment_id', commentId);

        return fetch(elyssaAiComments.ajax_url, {
            method: 'POST',
            credentials: 'same-origin',
            body: formData
        })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const ratingElement = document.getElementById(`elyssa-ai-rating-${commentId}`);
                    if (ratingElement) {
                        ratingElement.innerHTML = data.data.html;
                    }
                    return { success: true, commentId: commentId };
                } else {
                    return { success: false, commentId: commentId, error: data.data.message || 'Unknown error' };
                }
            })
            .catch(error => {
                return { success: false, commentId: commentId, error: error.message };
            })
            .finally(() => {
                if (typeof onComplete === 'function') {
                    onComplete();
                }
            });
    }

    function updateProgress(completed, total) {
        const progress = Math.round((completed / total) * 100);
        const rateButton = document.getElementById('elyssa-rate-comments');
        if (rateButton) {
            rateButton.textContent = `${elyssaAiComments.ratingInProgress}: ${progress}%`;
        }
    }

    function showLoadingIndicator() {
        if (!document.getElementById('elyssaLoadingIndicator')) {
            const indicator = document.createElement('div');
            indicator.id = 'elyssaLoadingIndicator';
            indicator.textContent = elyssaAiComments.loadingText;

            Object.assign(indicator.style, {
                position: 'fixed',
                top: '0',
                left: '0',
                width: '100%',
                height: '100%',
                backgroundColor: 'rgba(0, 0, 0, 0.5)',
                color: 'white',
                fontSize: '24px',
                display: 'flex',
                justifyContent: 'center',
                alignItems: 'center',
                zIndex: '9999'
            });

            document.body.appendChild(indicator);
        } else {
            document.getElementById('elyssaLoadingIndicator').style.display = 'flex';
        }
    }

    function hideLoadingIndicator() {
        const indicator = document.getElementById('elyssaLoadingIndicator');
        if (indicator) {
            indicator.style.display = 'none';
        }
    }

    function showAdminNotice(message, type) {
        const noticeDiv = document.createElement('div');
        noticeDiv.className = `notice notice-${type} is-dismissible`;
        noticeDiv.innerHTML = `<p>${message}</p>`;

        const wpHeaderEnd = document.getElementById('wpbody-content').getElementsByClassName('wp-header-end')[0];
        wpHeaderEnd.parentNode.insertBefore(noticeDiv, wpHeaderEnd.nextSibling);

        // Make the notice dismissible
        const dismissButton = document.createElement('button');
        dismissButton.type = 'button';
        dismissButton.className = 'notice-dismiss';
        dismissButton.innerHTML = '<span class="screen-reader-text">Dismiss this notice.</span>';
        dismissButton.onclick = function() {
            noticeDiv.parentNode.removeChild(noticeDiv);
        };
        noticeDiv.appendChild(dismissButton);
    }

    function setAdminNotice(message, type) {
        // Store the notice in sessionStorage
        sessionStorage.setItem('elyssaAiNotice', JSON.stringify({ message, type }));
    }

    function displayStoredNotice() {
        const storedNotice = sessionStorage.getItem('elyssaAiNotice');
        if (storedNotice) {
            const { message, type } = JSON.parse(storedNotice);
            showAdminNotice(message, type);
            sessionStorage.removeItem('elyssaAiNotice');
        }
    }

    if (document.readyState === 'complete' || document.readyState === 'interactive') {
        addRateButton();
        displayStoredNotice();
    } else {
        document.addEventListener('DOMContentLoaded', function() {
            addRateButton();
            displayStoredNotice();
        });
    }
})();