<?php
if (!defined('ABSPATH')) {
    exit;
}

class ElyssaAIEditor extends ElyssaAIBaseService
{
    public function __construct()
    {
        add_action('admin_enqueue_scripts', array($this, 'enqueue_editor_assets'));
        add_action('wp_ajax_elyssa_ai_rewrite', array($this, 'rewrite_ajax_handler'));
        add_action('wp_ajax_elyssa_ai_translate', array($this, 'translate_ajax_handler'));
        add_action('wp_ajax_elyssa_ai_new_article', array($this, 'new_article_ajax_handler'));
        add_action('wp_ajax_elyssa_ai_generate_media', array($this, 'generate_media_ajax_handler'));
        add_action('wp_ajax_elyssa_ai_generate_keywords', array($this, 'generate_keywords_ajax_handler'));
        add_action('add_meta_boxes', array($this, 'elyssa_ai_add_meta_box'));
        add_action('wp_ajax_elyssa_ai_get_prompt_content', array($this, 'get_prompt_content_ajax_handler'));
    }

    public function enqueue_editor_assets()
    {
        $screen = get_current_screen();
        if ($screen->post_type !== 'page' && $screen->post_type !== 'post') {
            return;
        }

        if ($this->detect_active_editor() == 'classic') {
            wp_enqueue_script(
                'elyssa-ai-editor-classic-js',
                plugin_dir_url(__FILE__) . 'js/elyssa-ai-editor-classic.js?'.time(),
                array('jquery'),
                time(),
                true
            );
            wp_set_script_translations('elyssa-ai-editor-classic-js', 'elyssa-ai');
        } else {
            wp_enqueue_script(
                'elyssa-ai-editor-gutenberg-js',
                plugin_dir_url(__FILE__) . 'js/elyssa-ai-editor-gutenberg.js?'.time(),
                array('wp-plugins', 'wp-edit-post', 'wp-element', 'wp-components', 'wp-data'),
                time(),
                true
            );
            wp_set_script_translations('elyssa-ai-editor-gutenberg-js', 'elyssa-ai');
        }

        $text_prompts = ElyssaAIPrompts::get_prompts_for_dropdown('text');
        $image_prompts = ElyssaAIPrompts::get_prompts_for_dropdown('image');

        wp_localize_script(
            $this->detect_active_editor() == 'classic' ? 'elyssa-ai-editor-classic-js' : 'elyssa-ai-editor-gutenberg-js',
            'elyssaAiEditorVars',
            array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('elyssa_ai_nonce'),
                'text_prompts' => $text_prompts,
                'image_prompts' => $image_prompts,
                'pluginUrl' => plugin_dir_url(dirname(__FILE__)),
                'version' => ELYSSA_AI_VERSION,
                'isTextEnabled' => elyssa_is_text_enabled(),
                'textDisabledText' => __('Please enable text services in the settings', 'elyssa-ai'),
                'isImageEnabled' => elyssa_is_image_enabled(),
                'imageDisabledText' => __('Please enable image services in the settings', 'elyssa-ai')
            )
        );
    }

    public function rewrite_ajax_handler()
    {
        check_ajax_referer('elyssa_ai_nonce', 'elyssa_ai_editor_nonce');

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'elyssa-ai')));
            return;
        }

        $init_result = $this->init_text_service();
        if (is_wp_error($init_result)) {
            wp_send_json_error(array('message' => $init_result->get_error_message()));
            return;
        }

        $content = isset($_POST['content']) ? wp_kses_post(wp_unslash($_POST['content'])) : '';

        if (empty($content)) {
            wp_send_json_error(array('message' => __('No content provided', 'elyssa-ai')));
            return;
        }

        $system_prompt = ServicePrompts::get_system('rewrite_text');
        $user_prompt = ServicePrompts::get_user('rewrite_text', $content);

        $rewritten_content = $this->text_service->generate_text($user_prompt, [], $system_prompt);

        if (is_wp_error($rewritten_content)) {
            wp_send_json_error(array('message' => $rewritten_content->get_error_message()));
        } else {
            $parsed_content = parse_structured_content($rewritten_content);
            wp_send_json_success($parsed_content);
        }
    }

    public function translate_ajax_handler()
    {
        check_ajax_referer('elyssa_ai_nonce', 'elyssa_ai_editor_nonce');

        $init_result = $this->init_text_service();
        if (is_wp_error($init_result)) {
            wp_send_json_error(array('message' => $init_result->get_error_message()));
            return;
        }

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'elyssa-ai')));
            return;
        }

        $content = isset($_POST['content']) ? wp_kses_post(wp_unslash($_POST['content'])) : '';
        $language = isset($_POST['language']) ? sanitize_text_field(wp_unslash($_POST['language'])) : '';

        if (empty($content) || empty($language)) {
            wp_send_json_error(array('message' => __('Content or language not provided', 'elyssa-ai')));
            return;
        }

        $system_prompt = ServicePrompts::get_system('translate_text');
        $user_prompt = sprintf(ServicePrompts::get_user('translate_text'), $language, $content);
        $translated_content = $this->text_service->generate_text($user_prompt, [], $system_prompt);

        if (is_wp_error($translated_content)) {
            wp_send_json_error(array('message' => $translated_content->get_error_message()));
        } else {
            $parsed_content = parse_structured_content($translated_content);
            wp_send_json_success($parsed_content);
        }
    }

    public function new_article_ajax_handler()
    {
        check_ajax_referer('elyssa_ai_nonce', 'elyssa_ai_editor_nonce');

        $init_result = $this->init_text_service();
        if (is_wp_error($init_result)) {
            wp_send_json_error(array('message' => $init_result->get_error_message()));
            return;
        }

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'elyssa-ai')));
            return;
        }

        $prompt = isset($_POST['prompt']) ? sanitize_text_field(wp_unslash($_POST['prompt'])) : '';

        if (empty($prompt)) {
            wp_send_json_error(array('message' => __('No prompt provided', 'elyssa-ai')));
            return;
        }

        $system_prompt = ServicePrompts::get_system('create_text');
        $user_prompt = ServicePrompts::get_user('create_text', $prompt);

        $article = $this->text_service->generate_text($user_prompt, [], $system_prompt);

        if (is_wp_error($article)) {
            wp_send_json_error(array('message' => $article->get_error_message()));
        } else {
            $parsed_content = parse_structured_content($article);
            wp_send_json_success($parsed_content);
        }
    }

    public function generate_media_ajax_handler()
    {
        check_ajax_referer('elyssa_ai_nonce', 'elyssa_ai_editor_nonce');

        $init_result = $this->init_image_service();
        if (is_wp_error($init_result)) {
            wp_send_json_error(array('message' => $init_result->get_error_message()));
            return;
        }

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'elyssa-ai')));
            return;
        }

        $prompt = isset($_POST['prompt']) ? sanitize_text_field(wp_unslash($_POST['prompt'])) : '';

        if (empty($prompt)) {
            wp_send_json_error(array('message' => __('No prompt provided', 'elyssa-ai')));
            return;
        }

        $image_url = $this->image_service->generate_image($prompt);

        if (is_wp_error($image_url)) {
            wp_send_json_error(array('message' => $image_url->get_error_message()));
        } else {
            $attachment_id = $this->upload_image_to_media_library($image_url, $prompt);
            if (is_wp_error($attachment_id)) {
                wp_send_json_error(array('message' => $attachment_id->get_error_message()));
            } else {
                $image_html = wp_get_attachment_image($attachment_id, 'full');
                wp_send_json_success(array('url' => wp_get_attachment_url($attachment_id), 'html' => $image_html));
            }
        }
    }

    public function generate_keywords_ajax_handler()
    {
        check_ajax_referer('elyssa_ai_nonce', 'elyssa_ai_editor_nonce');

        $init_result = $this->init_text_service();
        if (is_wp_error($init_result)) {
            wp_send_json_error(array('message' => $init_result->get_error_message()));
            return;
        }

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'elyssa-ai')));
            return;
        }

        $content = isset($_POST['content']) ? wp_kses_post(wp_unslash($_POST['content'])) : '';

        if (empty($content)) {
            wp_send_json_error(array('message' => __('No content provided', 'elyssa-ai')));
            return;
        }

        $system_prompt = ServicePrompts::get_system('generate_keywords');
        $user_prompt = sprintf(ServicePrompts::get_user('generate_keywords'), $content);

        $keywords = $this->text_service->generate_text($user_prompt, [], $system_prompt);

        if (is_wp_error($keywords)) {
            wp_send_json_error(array('message' => $keywords->get_error_message()));
        } else {
            $parsed_content = parse_structured_content($keywords);

            if (isset($parsed_content['keywords']) && !empty($parsed_content['keywords'])) {
                wp_send_json_success(array('keywords' => $parsed_content['keywords']));
            } else {
                wp_send_json_error(array('message' => __('No keywords generated', 'elyssa-ai')));
            }
        }
    }

    private function upload_image_to_media_library($image_url, $prompt)
    {
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');

        $tmp = download_url($image_url);
        if (is_wp_error($tmp)) {
            return $tmp;
        }

        $file_array = array(
            'name' => 'elyssa_ai_generated_image_' . time() . '.png',
            'tmp_name' => $tmp
        );

        $id = media_handle_sideload($file_array, 0);

        if (is_wp_error($id)) {
            @wp_delete_file($file_array['tmp_name']);
            return $id;
        }

        wp_update_post(array(
            'ID' => $id,
            'post_excerpt' => sanitize_text_field($prompt),
            'post_content' => sanitize_text_field($prompt)
        ));

        update_post_meta($id, '_wp_attachment_image_alt', sanitize_text_field($prompt));

        return $id;
    }

    private function detect_active_editor() {
        // Check if we're in the admin area and editing a post/page
        if (!is_admin() || !function_exists('get_current_screen')) {
            return false;
        }

        $screen = get_current_screen();
        if (!$screen || !in_array($screen->base, ['post', 'page'])) {
            return false;
        }

        if(method_exists( $screen, 'is_block_editor' ) && $screen->is_block_editor()) {
            return 'block';
        }

        return 'classic';
    }

    public function elyssa_ai_add_meta_box()
    {
        if ($this->detect_active_editor() === 'classic') {
            $post_types = array('post', 'page');
            foreach ($post_types as $post_type) {
                add_meta_box(
                    'elyssa_ai_metabox',
                    'elyssa AI',
                    array($this, 'elyssa_ai_metabox_callback'),
                    $post_type,
                    'side',
                    'default'
                );
            }
        }
    }

    public function elyssa_ai_metabox_callback($post)
    {
        wp_nonce_field('elyssa_ai_metabox', 'elyssa_ai_nonce');
        $text_prompts = ElyssaAIPrompts::get_prompts_for_dropdown('text');
        $image_prompts = ElyssaAIPrompts::get_prompts_for_dropdown('image');
        
        // Registriere und enqueue Metabox-Styles
        wp_register_style('elyssa-ai-metabox-styles', false);
        wp_enqueue_style('elyssa-ai-metabox-styles');
        
        // Füge Inline-CSS hinzu
        $metabox_css = "
            #elyssa-ai-container { text-align: center; }
            #elyssa-ai-container .logo-container { display: flex; align-items: center; margin: 0; margin-bottom: 10px; }
            #elyssa-ai-container .logo-wrapper { margin-top: 10px; }
            #elyssa-ai-container .control-section { width: 90%; margin: 0 auto; }
            #elyssa-ai-container select { width: 100%; margin-bottom: 5px; }
            #elyssa-ai-container textarea { width: 100%; }
            #elyssa-ai-container .elyssa-ai-button { width: 100%; }
            #elyssa-ai-container .translate-button { margin-top: 5px; }
        ";
        wp_add_inline_style('elyssa-ai-metabox-styles', $metabox_css);
        
        // Registriere und enqueue Metabox-Initialisierungs-Skript
        wp_register_script('elyssa-ai-metabox-init', '', array('jquery'), ELYSSA_AI_VERSION, true);
        
        // Bereite die Dropdown-Optionen für Prompts vor
        $text_prompt_options = '';
        foreach ($text_prompts as $id => $title) {
            $text_prompt_options .= '<option value="' . esc_attr($id) . '">' . esc_html($title) . '</option>';
        }
        
        $image_prompt_options = '';
        foreach ($image_prompts as $id => $title) {
            $image_prompt_options .= '<option value="' . esc_attr($id) . '">' . esc_html($title) . '</option>';
        }
        
        // Erstelle die Sprachoptionen für die Übersetzungsfunktion
        $language_options = '
            <option value="">' . esc_html__('Select Language', 'elyssa-ai') . '</option>
            <option value="de">' . esc_html__('German', 'elyssa-ai') . '</option>
            <option value="en">' . esc_html__('English', 'elyssa-ai') . '</option>
            <option value="fr">' . esc_html__('French', 'elyssa-ai') . '</option>
            <option value="es">' . esc_html__('Spanish', 'elyssa-ai') . '</option>
            <option value="it">' . esc_html__('Italian', 'elyssa-ai') . '</option>
            <option value="nl">' . esc_html__('Dutch', 'elyssa-ai') . '</option>
            <option value="pt">' . esc_html__('Portuguese', 'elyssa-ai') . '</option>
            <option value="pl">' . esc_html__('Polish', 'elyssa-ai') . '</option>
            <option value="se">' . esc_html__('Swedish', 'elyssa-ai') . '</option>
            <option value="tr">' . esc_html__('Turkish', 'elyssa-ai') . '</option>
            <option value="cn">' . esc_html__('Chinese', 'elyssa-ai') . '</option>
            <option value="jp">' . esc_html__('Japanese', 'elyssa-ai') . '</option>
        ';
        
        // Text für die Buttons und disabled/title Status
        $text_disabled = !elyssa_is_text_enabled() ? 'disabled' : '';
        $text_disabled_title = !elyssa_is_text_enabled() ? 'title="' . esc_attr__('Please enable text services in the settings', 'elyssa-ai') . '"' : '';
        
        $image_disabled = !elyssa_is_image_enabled() ? 'disabled' : '';
        $image_disabled_title = !elyssa_is_image_enabled() ? 'title="' . esc_attr__('Please enable image services in the settings', 'elyssa-ai') . '"' : '';
        
        // Erstelle den HTML-Output-Container
        echo '<div id="elyssa-ai-container"></div>';
        
        // Das JavaScript, das das Metabox-HTML erstellt
        $init_script = "jQuery(document).ready(function($) {";
        $init_script .= "  const container = $('#elyssa-ai-container');";
        
        $init_script .= "  let metaboxHtml = `";
        
        // Logo-Sektion
        $init_script .= "    <div class='logo-container'>";
        $init_script .= "      <div class='logo-wrapper'>";
        $init_script .= "        <img src='" . esc_js(plugins_url('elyssa-ai/images/elyssa-logo-h100.webp')) . "' alt='elyssa AI Logo' style='height: 52px;'>";
        $init_script .= "      </div>";
        $init_script .= "    </div>";
        
        // Text-Prompts und Artikelerstellung
        $init_script .= "    <div class='control-section'>";
        if (!empty($text_prompts)) {
            $init_script .= "    <select id='elyssa-ai-text-prompt-select'>";
            $init_script .= "      <option value=''>" . esc_js(__('Select a text prompt', 'elyssa-ai')) . "</option>";
            $init_script .= "      $text_prompt_options";
            $init_script .= "    </select>";
        }
        $init_script .= "      <p>";
        $init_script .= "        <textarea id='elyssa-ai-new-article-prompt' placeholder='" . esc_js(__('New article prompt', 'elyssa-ai')) . "' rows='5'></textarea>";
        $init_script .= "        <button type='button' id='elyssa-ai-new-article' class='elyssa-ai-button' $text_disabled $text_disabled_title>" . esc_js(__('Create New Article', 'elyssa-ai')) . "</button>";
        $init_script .= "      </p>";
        $init_script .= "    </div>";
        
        // Schreibfunktion
        $init_script .= "    <hr/>";
        $init_script .= "    <div class='control-section'>";
        $init_script .= "      <p>";
        $init_script .= "        <button type='button' id='elyssa-ai-rewrite' class='elyssa-ai-button' $text_disabled $text_disabled_title>" . esc_js(__('Rewrite Content', 'elyssa-ai')) . "</button>";
        $init_script .= "      </p>";
        $init_script .= "    </div>";
        
        // Übersetzungsfunktion
        $init_script .= "    <hr/>";
        $init_script .= "    <div class='control-section'>";
        $init_script .= "      <p>";
        $init_script .= "        <select id='elyssa-ai-translate-language'>";
        $init_script .= "          $language_options";
        $init_script .= "        </select>";
        $init_script .= "        <button type='button' id='elyssa-ai-translate' class='elyssa-ai-button translate-button' $text_disabled $text_disabled_title>" . esc_js(__('Translate', 'elyssa-ai')) . "</button>";
        $init_script .= "      </p>";
        $init_script .= "    </div>";
        
        // Bilder-Generator
        $init_script .= "    <hr/>";
        $init_script .= "    <div class='control-section'>";
        if (!empty($image_prompts)) {
            $init_script .= "    <select id='elyssa-ai-image-prompt-select'>";
            $init_script .= "      <option value=''>" . esc_js(__('Select an image prompt', 'elyssa-ai')) . "</option>";
            $init_script .= "      $image_prompt_options";
            $init_script .= "    </select>";
        }
        $init_script .= "      <p>";
        $init_script .= "        <textarea id='elyssa-ai-image-prompt' placeholder='" . esc_js(__('Image generation prompt', 'elyssa-ai')) . "' rows='3'></textarea>";
        $init_script .= "        <button type='button' id='elyssa-ai-generate-media' class='elyssa-ai-button' $image_disabled $image_disabled_title>" . esc_js(__('Generate Media', 'elyssa-ai')) . "</button>";
        $init_script .= "      </p>";
        $init_script .= "    </div>";
        
        // Keywords-Generator (nur für Posts und Produkte)
        if ($post->post_type === 'post' || $post->post_type === 'product') {
            $init_script .= "  <hr/>";
            $init_script .= "  <div class='control-section'>";
            $init_script .= "    <p>";
            $init_script .= "      <button type='button' id='elyssa-ai-generate-keywords' class='elyssa-ai-button' $text_disabled $text_disabled_title>" . esc_js(__('Generate Keywords', 'elyssa-ai')) . "</button>";
            $init_script .= "    </p>";
            $init_script .= "  </div>";
        }
        
        $init_script .= "  `;
        
        ";
        $init_script .= "  container.html(metaboxHtml);";
        $init_script .= "});";
        
        wp_add_inline_script('elyssa-ai-metabox-init', $init_script);
        wp_enqueue_script('elyssa-ai-metabox-init');
    }
}

// Instantiate the class
new ElyssaAIEditor();