(function (wp) {
    var registerPlugin = wp.plugins.registerPlugin;
    var PluginSidebar = wp.editPost.PluginSidebar;
    var PluginSidebarMoreMenuItem = wp.editPost.PluginSidebarMoreMenuItem;
    var el = wp.element.createElement;
    var Button = wp.components.Button;
    var TextareaControl = wp.components.TextareaControl;
    var SelectControl = wp.components.SelectControl;
    var useSelect = wp.data.useSelect;
    var useDispatch = wp.data.useDispatch;
    var apiFetch = wp.apiFetch;
    const { useState, useEffect } = wp.element;
    const { __ } = wp.i18n;

    function ElyssaAIIcon() {
        return el('svg', {width: 20, height: 20, viewBox: "0 0 20 20"},
            el('path', {d: "M9.813 15.904 9 18.75l-.813-2.846a4.5 4.5 0 0 0-3.09-3.09L2.25 12l2.846-.813a4.5 4.5 0 0 0 3.09-3.09L9 5.25l.813 2.846a4.5 4.5 0 0 0 3.09 3.09L15.75 12l-2.846.813a4.5 4.5 0 0 0-3.09 3.09ZM18.259 8.715 18 9.75l-.259-1.035a3.375 3.375 0 0 0-2.455-2.456L14.25 6l1.036-.259a3.375 3.375 0 0 0 2.455-2.456L18 2.25l.259 1.035a3.375 3.375 0 0 0 2.456 2.456L21.75 6l-1.035.259a3.375 3.375 0 0 0-2.456 2.456ZM16.894 20.567 16.5 21.75l-.394-1.183a2.25 2.25 0 0 0-1.423-1.423L13.5 18.75l1.183-.394a2.25 2.25 0 0 0 1.423-1.423l.394-1.183.394 1.183a2.25 2.25 0 0 0 1.423 1.423l1.183.394-1.183.394a2.25 2.25 0 0 0-1.423 1.423Z"})
        );
    }

    const PromptSelect = ({ onSelect, type }) => {
        const [prompts, setPrompts] = useState([]);
        const [error, setError] = useState(null);

        useEffect(() => {
            const promptsData = type === 'text' ? elyssaAiEditorVars.text_prompts : elyssaAiEditorVars.image_prompts;
            if (promptsData) {
                const promptOptions = Object.entries(promptsData).map(([id, title]) => ({ value: id, label: title }));
                setPrompts(promptOptions);
            } else {
                setError(`No ${type} prompts available`);
            }
        }, [type]);


        const handleChange = (selectedId) => {
            setError(null);
            if (selectedId) {
                wp.apiFetch({
                    path: '/wp/v2/elyssa_ai_prompt/' + selectedId,
                    method: 'GET',
                }).then(prompt => {
                    console.log("Fetched prompt:", prompt);
                    if (prompt && prompt.meta && prompt.meta._user_prompt) {
                        onSelect(prompt.meta._user_prompt);
                    } else {
                        setError("Unexpected prompt structure");
                    }
                }).catch(error => {
                    setError("Error fetching prompt");
                    console.error("Error fetching prompt:", error);
                });
            }
        };

        if (error) {
            return el('p', { style: { color: 'red' } }, error);
        }

        if (prompts.length === 0) {
            return el('p', {}, __('No prompts available', 'elyssa-ai'));
        }

        return el(SelectControl, {
            label: __('Select a prompt', 'elyssa-ai'),
            options: [{ value: '', label: __('Select a prompt', 'elyssa-ai') }, ...prompts],
            onChange: handleChange
        });
    };

    function showLoadingIndicator() {
        const loadingOverlay = document.getElementById('elyssaLoadingOverlay');
        if (!loadingOverlay) {
            const newOverlay = document.createElement('div');
            newOverlay.id = 'elyssaLoadingOverlay';
            newOverlay.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background-color: rgba(0, 0, 0, 0.5);
                display: flex;
                justify-content: center;
                align-items: center;
                z-index: 9999;
            `;
            const loadingText = document.createElement('div');
            loadingText.textContent = wp.i18n.__('Loading...', 'elyssa-ai');
            loadingText.style.cssText = `
                color: white;
                font-size: 24px;
            `;
            newOverlay.appendChild(loadingText);
            document.body.appendChild(newOverlay);
        } else {
            loadingOverlay.style.display = 'flex';
        }
    }

    function hideLoadingIndicator() {
        const loadingOverlay = document.getElementById('elyssaLoadingOverlay');
        if (loadingOverlay) {
            loadingOverlay.style.display = 'none';
        }
    }

    function elyssaAjaxRequest(action, data) {
        showLoadingIndicator();
        console.log(action, data);
        return fetch(elyssaAiEditorVars.ajax_url, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: new URLSearchParams({
                action: 'elyssa_ai_' + action,
                elyssa_ai_editor_nonce: elyssaAiEditorVars.nonce,
                ...data
            })
        })
            .then(response => response.json())
            .then(data => {
                hideLoadingIndicator();
                if (data.success) {
                    return data.data;
                } else {
                    throw new Error(data.data.message || 'Unknown error occurred');
                }
            })
            .catch(error => {
                hideLoadingIndicator();
                console.error('Error:', error);
                throw error;
            });
    }

    function ElyssaAISidebar() {
        const [targetLanguage, setTargetLanguage] = useState('');
        const [newArticlePrompt, setNewArticlePrompt] = useState('');
        const [imagePrompt, setImagePrompt] = useState('');

        const { createNotice } = useDispatch('core/notices');
        const { editPost } = useDispatch('core/editor');

        const postContent = useSelect(select =>
            select('core/editor').getEditedPostContent()
        );

        const postType = useSelect(select =>
            select('core/editor').getCurrentPostType()
        );

        const currentTags = useSelect(select =>
            select('core/editor').getEditedPostAttribute('tags') || []
        );

        const handlePromptSelect = (prompt) => {
            console.log("Selected prompt:", prompt);
            setNewArticlePrompt(prompt);
        };

        const handleImagePromptSelect = (prompt) => {
            setImagePrompt(prompt);
        };

        function updateEditorContent(title, content) {
            wp.data.dispatch('core/editor').editPost({
                title: title,
                content: content
            });
        }

        const languageOptions = [
            {label: wp.i18n.__('Select Language', 'elyssa-ai'), value: ''},
            {label: wp.i18n.__('German', 'elyssa-ai'), value: 'de'},
            {label: wp.i18n.__('English', 'elyssa-ai'), value: 'en'},
            {label: wp.i18n.__('French', 'elyssa-ai'), value: 'fr'},
            {label: wp.i18n.__('Spanish', 'elyssa-ai'), value: 'es'},
            {label: wp.i18n.__('Italian', 'elyssa-ai'), value: 'it'},
            {label: wp.i18n.__('Dutch', 'elyssa-ai'), value: 'nl'},
            {label: wp.i18n.__('Portuguese', 'elyssa-ai'), value: 'pt'},
            {label: wp.i18n.__('Polish', 'elyssa-ai'), value: 'pl'},
            {label: wp.i18n.__('Swedish', 'elyssa-ai'), value: 'se'},
            {label: wp.i18n.__('Turkish', 'elyssa-ai'), value: 'tr'},
            {label: wp.i18n.__('Chinese (simplified)', 'elyssa-ai'), value: 'cn'},
            {label: wp.i18n.__('Japanese', 'elyssa-ai'), value: 'jp'},
        ];

        function rewriteContent() {
            var content = wp.data.select('core/editor').getEditedPostContent();

            elyssaAjaxRequest('rewrite', { content: content })
                .then(result => {
                    if (result.title && result.content) {
                        updateEditorContent(result.title, result.content);
                    }
                    if (result.keywords) {
                        updateKeywords(result.keywords);
                    }
                })
                .catch(error => {
                    console.error('Rewrite error:', error);
                    createNotice('error', __('An error occurred while rewriting: ', 'elyssa-ai') + error.message);
                });
        }

        function translateContent() {
            if (!targetLanguage) {
                createNotice('error', __('Please select a target language', 'elyssa-ai'));
                return;
            }

            var content = wp.data.select('core/editor').getEditedPostContent();

            elyssaAjaxRequest('translate', { content: content, language: targetLanguage })
                .then(result => {
                    if (result.title && result.content) {
                        updateEditorContent(result.title, result.content);
                    }
                    if (result.keywords) {
                        updateKeywords(result.keywords);
                    }
                })
                .catch(error => {
                    console.error('Translate error:', error);
                    createNotice('error', __('An error occurred while translating: ', 'elyssa-ai') + error.message);
                });
        }

        function createNewArticle() {
            if (!newArticlePrompt) {
                createNotice('error', __('Please enter a prompt for the new article', 'elyssa-ai'));
                return;
            }

            elyssaAjaxRequest('new_article', { prompt: newArticlePrompt })
                .then(result => {
                    if (result.title && result.content) {
                        updateEditorContent(result.title, result.content);
                    }
                    if (result.keywords) {
                        updateKeywords(result.keywords);
                    }
                })
                .catch(error => {
                    console.error('New article error:', error);
                    createNotice('error', __('An error occurred while creating a new article: ', 'elyssa-ai') + error.message);
                });
        }

        function generateMedia() {
            if (!imagePrompt) {
                createNotice('error', __('Please enter an image prompt', 'elyssa-ai'));
                return;
            }

            elyssaAjaxRequest('generate_media', { prompt: imagePrompt })
                .then(result => {
                    if (result.url) {
                        const block = wp.blocks.createBlock('core/image', {
                            url: result.url,
                            caption: imagePrompt
                        });
                        wp.data.dispatch('core/block-editor').insertBlocks(block);
                        createNotice('success', __('Image has been generated and inserted into the post.', 'elyssa-ai'));
                    }
                })
                .catch(error => {
                    console.error('Generate media error:', error);
                    createNotice('error', __('An error occurred while generating media: ', 'elyssa-ai') + error.message);
                });
        }

        function generateKeywords() {
            elyssaAjaxRequest('generate_keywords', { content: postContent })
                .then(result => {
                    if (result.keywords) {
                        updateKeywords(result.keywords);
                        createNotice('success', __('Keywords have been generated and added to the post tags.', 'elyssa-ai'));
                    }
                })
                .catch(error => {
                    console.error('Generate keywords error:', error);
                    createNotice('error', __('An error occurred while generating keywords: ', 'elyssa-ai') + error.message);
                });
        }

        function updateKeywords(keywords) {
            if (Array.isArray(keywords)) {
                const newTagIds = [];
                const existingTags = wp.data.select('core').getEntityRecords('taxonomy', 'post_tag', { per_page: -1 }) || [];

                keywords.forEach(keyword => {
                    const existingTag = existingTags.find(tag => tag.name.toLowerCase() === keyword.toLowerCase());
                    if (existingTag) {
                        newTagIds.push(existingTag.id);
                    } else {
                        // Create a new tag
                        wp.apiFetch({
                            path: '/wp/v2/tags',
                            method: 'POST',
                            data: { name: keyword }
                        }).then(newTag => {
                            newTagIds.push(newTag.id);
                        });
                    }
                });

                // Update the post with new tags
                wp.data.dispatch('core/editor').editPost({ tags: newTagIds });
            }
        }

        return [
            el(
                PluginSidebarMoreMenuItem,
                {
                    target: 'elyssa-ai-sidebar',
                    icon: ElyssaAIIcon
                },
                __('elyssa AI')
            ),
            el(
                PluginSidebar,
                {
                    name: 'elyssa-ai-sidebar',
                    title: __('elyssa AI'),
                    icon: ElyssaAIIcon,
                },
                el(
                    'div',
                    {
                        className: 'elyssa-ai-sidebar-content',
                        style: {
                            padding: '16px',
                        }
                    },
                    el('div',
                        {
                            style: {
                                display: 'flex',
                                alignItems: 'center',
                                marginBottom: '16px',
                            }
                        },
                        el('img', {
                            src: elyssaAiEditorVars.pluginUrl + '/images/elyssa-logo-h100.webp',
                            alt: 'elyssa AI Logo',
                            style: {
                                height: '52px',
                                marginRight: '10px',
                            }
                        }),
                    ),

                    el(PromptSelect, {
                        onSelect: handlePromptSelect,
                        type: 'text'
                    }),
                    el(
                        TextareaControl,
                        {
                            label: __('New article'),
                            value: newArticlePrompt,
                            onChange: setNewArticlePrompt,
                            placeholder: __('Prompt here'),
                            rows: 5
                        }
                    ),
                    el(
                        Button,
                        {
                            as: 'button',
                            onClick: createNewArticle,
                            disabled: !elyssaAiEditorVars.isTextEnabled,
                            title: !elyssaAiEditorVars.isTextEnabled ? elyssaAiEditorVars.textDisabledText : '',
                            style: {
                                width: '100%',
                                justifyContent: 'center',
                                marginTop: '8px',
                                marginBottom: '16px'
                            },
                            className: 'elyssa-ai-button'
                        },
                        __('Create New Article')
                    ),
                    el('hr'),
                    el(
                        Button,
                        {
                            as: 'button',
                            onClick: rewriteContent,
                            disabled: !elyssaAiEditorVars.isTextEnabled,
                            title: !elyssaAiEditorVars.isTextEnabled ? elyssaAiEditorVars.textDisabledText : '',
                            style: {
                                width: '100%',
                                justifyContent: 'center',
                                marginTop: '16px',
                                marginBottom: '16px'
                            },
                            className: 'elyssa-ai-button'
                        },
                        __('Rewrite article')
                    ),
                    el('hr'),
                    el(
                        SelectControl,
                        {
                            label: __('Target language'),
                            value: targetLanguage,
                            options: languageOptions,
                            onChange: setTargetLanguage
                        }
                    ),
                    el(
                        Button,
                        {
                            as: 'button',
                            onClick: translateContent,
                            disabled: !elyssaAiEditorVars.isTextEnabled,
                            title: !elyssaAiEditorVars.isTextEnabled ? elyssaAiEditorVars.textDisabledText : '',
                            style: {
                                width: '100%',
                                justifyContent: 'center',
                                marginTop: '8px',
                                marginBottom: '16px'
                            },
                            className: 'elyssa-ai-button'
                        },
                        __('Translate article')
                    ),
                    el('hr'),
                    el(PromptSelect, {
                        onSelect: handleImagePromptSelect,
                        type: 'image'
                    }),
                    el(
                        TextareaControl,
                        {
                            label: __('Image prompt'),
                            value: imagePrompt,
                            onChange: setImagePrompt,
                            placeholder: __('Describe the image you want to generate'),
                            rows: 3
                        }
                    ),
                    el(
                        Button,
                        {
                            as: 'button',
                            onClick: generateMedia,
                            disabled: !elyssaAiEditorVars.isImageEnabled,
                            title: !elyssaAiEditorVars.isImageEnabled ? elyssaAiEditorVars.imageDisabledText : '',
                            style: {
                                width: '100%',
                                justifyContent: 'center',
                                marginTop: '8px'
                            },
                            className: 'elyssa-ai-button',
                        },
                        __('Generate Media')
                    ),
                    el('hr'),
                    postType === 'post' && el(
                        Button,
                        {
                            as: 'button',
                            onClick: generateKeywords,
                            disabled: !elyssaAiEditorVars.isTextEnabled,
                            title: !elyssaAiEditorVars.isTextEnabled ? elyssaAiEditorVars.textDisabledText : '',
                            style: {
                                width: '100%',
                                justifyContent: 'center',
                                marginTop: '8px',
                                marginBottom: '16px'
                            },
                            className: 'elyssa-ai-button'
                        },
                        __('Generate Keywords')
                    )
                )
            )
        ];
    }

    registerPlugin('elyssa-ai-sidebar', {
        render: ElyssaAISidebar,
        icon: ElyssaAIIcon
    });
})(window.wp);