<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * Factory class for managing AI service instances
 */
class ElyssaAI_ServiceFactory {
    private static $text_services = [];
    private static $image_services = [];

    /**
     * Register a text service
     *
     * @param string $name Service identifier
     * @param string $class_name Class name of the service
     */
    public static function register_text_service($name, $class_name) {
        self::$text_services[$name] = $class_name;
    }

    /**
     * Register an image service
     * 
     * @param string $name Service identifier
     * @param string $class_name Class name of the service
     */
    public static function register_image_service($name, $class_name) {
        self::$image_services[$name] = $class_name;
    }

    /**
     * Get an instance of a text service
     *
     * @param string $name Service identifier
     * @return ElyssaAI_Service Service instance
     * @throws Exception If service is not found
     */
    public static function get_text_service($name) {
        if (!isset(self::$text_services[$name])) {
            throw new Exception(esc_html__("Unknown text AI service: ", 'elyssa-ai') . esc_html($name));
        }
        $class_name = self::$text_services[$name];
        $model = get_option('elyssa_ai_text_model', ''); // Default model, if not set
        return new $class_name('text', $model);
    }

    /**
     * Get an instance of an image service
     *
     * @param string $name Service identifier
     * @return ElyssaAI_Service Service instance
     * @throws Exception If service is not found
     */
    public static function get_image_service($name) {
        if (!isset(self::$image_services[$name])) {
            throw new Exception(esc_html__("Unknown image AI service: ", 'elyssa-ai') . esc_html($name));
        }
        $class_name = self::$image_services[$name];
        $model = get_option('elyssa_ai_image_model', ''); // Default model, if not set
        return new $class_name('image', $model);
    }

    /**
     * Get a service instance by type
     *
     * @param string $type Service type ('text' or 'image')
     * @param string|null $model Optional model override
     * @return ElyssaAI_Service Service instance
     * @throws Exception If no service is configured
     */
    public static function get_service($type, $model = null) {
        $service_name = get_option("elyssa_ai_{$type}_service");
        $selected_model = $model ?? get_option("elyssa_ai_{$type}_model");

        if (!$service_name) {
            throw new Exception(esc_html__("No AI service configured for ", 'elyssa-ai') . esc_html($type));
        }

        $class_name = self::${$type . '_services'}[$service_name];
        return new $class_name($type, $selected_model);
    }

    /**
     * Get available models for a service type
     * 
     * Note: This method is no longer used as models are now manually entered.
     * The function is kept for backward compatibility but returns an empty array.
     *
     * @param string $type Service type ('text' or 'image')
     * @return array Empty array as models are now manually entered
     */
    public static function get_available_models($type) {
        return [];
    }

    /**
     * Get all available services
     *
     * @return array Available services by type
     */
    public static function get_available_services() {
        return [
            'text' => self::$text_services,
            'image' => self::$image_services
        ];
    }
}
