<?php
// Prevent direct access to this file
if (!defined('ABSPATH')) {
    exit;
}

class ElyssaAIMedia extends ElyssaAIBaseService
{
    public function __construct() {
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('admin_footer', array($this, 'add_modal_html'));
        add_action('load-upload.php', array($this, 'media_generate_index'));
        add_action('post-plupload-upload-ui', array($this, 'media_generate_media'));
        add_filter('media_buttons', array($this, 'add_media_button'), 15);

        add_action('wp_ajax_elyssa_ai_media_request', array($this, 'handle_ajax_request'));
        add_action('wp_ajax_elyssa_ai_media_save_request', array($this, 'handle_save_ajax_request'));
        add_action('wp_ajax_elyssa_ai_media_get_prompt_content', array($this, 'get_prompt_content_ajax_handler'));
    }

    public function enqueue_scripts($hook) {
        if (!in_array($hook, array('upload.php', 'post.php', 'post-new.php'))) {
            return;
        }

        wp_enqueue_script('elyssa-ai-media-js', plugin_dir_url(__FILE__) . 'assets/elyssa-media.js?'.time(), array('jquery'), ELYSSA_AI_VERSION, true);
        wp_set_script_translations( 'elyssa-ai-media-js', 'elyssa-ai' );

        wp_enqueue_style('elyssa-ai-media-css', plugin_dir_url(__FILE__) . 'assets/elyssa-media.css', array(), ELYSSA_AI_VERSION);

        $media_nonce = wp_create_nonce('elyssa_ai_media_nonce');
        $prompt_nonce = wp_create_nonce('elyssa_ai_prompt_nonce');

        wp_localize_script('elyssa-ai-media-js', 'elyssaAiMediaVars', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'media_nonce' => $media_nonce,
            'prompt_nonce' => $prompt_nonce,
            'can_upload' => current_user_can('upload_files'),
            'is_media_library' => $hook === 'upload.php',
        ));
    }

    public function add_modal_html() {
        $screen = get_current_screen();
        if (!in_array($screen->id, array('upload', 'post', 'page'))) {
            return;
        }

        $is_media_library = $screen->id === 'upload';
        $save_button_text = $is_media_library ? __('Save to Media Library', 'elyssa-ai') : __('Save to Media Library and Insert', 'elyssa-ai');
        $image_prompts = ElyssaAIPrompts::get_prompts_for_dropdown('image');
        ?>
        <div id="elyssa-ai-media-modal" class="elyssa-ai-media-modal">
            <div class="elyssa-ai-media-modal-content">
                <span class="elyssa-ai-media-close -mt-4">&times;</span>
                <div style="display: flex; align-items: center; margin: 0;margin-bottom: 10px;">
                    <div style="margin-top: 10px;margin: 0 auto;">
                        <img src="<?php echo esc_url(plugins_url('elyssa-ai/images/elyssa-logo-h100.webp')) ?>" alt="elyssa AI Logo" style="height: 60px;">
                    </div>
                </div>
                <?php if (!empty($image_prompts)) : ?>
                    <select id="elyssa-ai-media-prompt-select" class="w-full mb-2 mb-3 mt-4" style="width:100%">
                        <option value=""><?php esc_html_e('Select an image prompt', 'elyssa-ai'); ?></option>
                        <?php foreach ($image_prompts as $id => $title) : ?>
                            <option value="<?php echo esc_attr($id); ?>"><?php echo esc_html($title); ?></option>
                        <?php endforeach; ?>
                    </select>
                <?php endif; ?>
                <textarea id="elyssa-ai-media-prompt" rows="4" placeholder="<?php esc_attr_e('Enter your image prompt here', 'elyssa-ai'); ?>" class="p-2"></textarea>
                <button id="elyssa-ai-media-generate"
                        class="elyssa-ai-button"
                        <?php if(!elyssa_is_image_enabled()): ?>title="<?php esc_html_e('Please enable text services in the settings', 'elyssa-ai'); ?>"<?php endif;?>
                        <?php if(!elyssa_is_image_enabled()): ?>disabled<?php endif;?>
                ><?php esc_html_e('Generate Image', 'elyssa-ai'); ?></button>
                <div id="elyssa-ai-media-result"></div>
                <button id="elyssa-ai-media-save" class="elyssa-ai-button" style="display:none;"><?php echo esc_html($save_button_text); ?></button>
            </div>
        </div>
        <?php
    }

    public function handle_ajax_request() {
        check_ajax_referer('elyssa_ai_media_nonce', 'nonce');

        if (!current_user_can('upload_files')) {
            wp_send_json_error(array('message' => __('You do not have permission to upload files.', 'elyssa-ai')));
            return;
        }

        $init_result = $this->init_image_service();
        if (is_wp_error($init_result)) {
            wp_send_json_error(array('message' => $init_result->get_error_message()));
            return;
        }

        $prompt = sanitize_text_field(wp_unslash($_POST['prompt'] ?? ''));
        elyssa_ai_log($prompt);

        if (!$this->image_service) {
            wp_send_json_error(array('message' => __('Image AI service not available.', 'elyssa-ai')));
            return;
        }

        try {
            $image_url = $this->image_service->generate_image($prompt);
            elyssa_ai_log($image_url);

            wp_send_json_success([
                'message' => '',
                'image' => $image_url,
            ]);
        } catch (Exception $e) {
            wp_send_json_error(array('message' => __('Error generating image:', 'elyssa-ai') . ' ' . $e->getMessage()));
        }
    }

    public function handle_save_ajax_request() {
        check_ajax_referer('elyssa_ai_media_nonce', 'nonce');

        elyssa_ai_log(__("Save request received", 'elyssa-ai'));

        if (!current_user_can('upload_files')) {
            elyssa_ai_log(__("User does not have permission to upload files", 'elyssa-ai'));
            wp_send_json_error(array('message' => __('You do not have permission to upload files.', 'elyssa-ai')));
            return;
        }

        $image_url = esc_url_raw(wp_unslash($_POST['image'] ?? ''));
        $prompt = sanitize_textarea_field(wp_unslash($_POST['prompt'] ?? ''));

        elyssa_ai_log(__("Image URL: ", 'elyssa-ai') . $image_url);
        elyssa_ai_log(__("Prompt: ", 'elyssa-ai') . $prompt);

        if ($image_url === '') {
            elyssa_ai_log(__("No image URL provided", 'elyssa-ai'));
            wp_send_json_error([
                'message' => __('No image to save', 'elyssa-ai'),
            ]);
        }

        $attachment_id = $this->download_image_to_media_library($image_url, 0, $prompt);

        if ($attachment_id) {
            elyssa_ai_log(__("Image saved successfully. Attachment ID: ", 'elyssa-ai') . $attachment_id);
            $attachment_url = wp_get_attachment_url($attachment_id);
            $attachment_data = wp_get_attachment_metadata($attachment_id);

            wp_send_json_success([
                'message' => __('Image saved in media library', 'elyssa-ai'),
                'attachment_id' => $attachment_id,
                'url' => $attachment_url,
                'width' => $attachment_data['width'] ?? '',
                'height' => $attachment_data['height'] ?? '',
                'description' => $prompt
            ]);
        } else {
            elyssa_ai_log(__("Failed to save image in media library", 'elyssa-ai'));
            wp_send_json_error([
                'message' => __('Failed to save image in media library. Please check server logs for more details.', 'elyssa-ai'),
            ]);
        }
    }

    public function media_generate_index() {
        if (!current_user_can('upload_files')) return;
        add_filter('esc_html', array($this, 'media_generate_title'), 999, 2);
    }

    public function media_generate_title($btn, $text) {
        if (!current_user_can('upload_files')) return $btn;
        if ($text === __('Media Library', 'elyssa-ai') && did_action('all_admin_notices')) {
            remove_filter('esc_html', array($this, 'media_generate_title'), 999, 2);
            $elyssa_generate = '<button class="elyssa-ai-button" style="margin-left:30px;" id="elyssa-ai-media-open-modal" title="'.(!elyssa_is_image_enabled() ? __('Please enable image services in the settings', 'elyssa-ai') : '').'" '.(!elyssa_is_image_enabled() ? 'disabled' : '').'><img src="'.esc_url(plugins_url('elyssa-ai/images/elyssa-icon-white.webp')) .'" alt="elyssa AI Logo" style="width:15px;height:15px;float:left;margin-right:6px;">' . esc_html__('elyssa AI Media', 'elyssa-ai') . '</button>';
            $btn = $btn . $elyssa_generate;
        }
        return $btn;
    }

    public function media_generate_media() {
        if (preg_match('/media|post|upload/i', basename(get_permalink())))
            if (current_user_can('upload_files')) {
                echo '<div><p align="center">';
                echo '<input id="elyssa-ai-media-open-modal" type="button" value="' . esc_attr__('elyssa Generate Image', 'elyssa-ai') . '" class="elyssa-ai-button" />';
                echo '</p></div>';
            }
    }

    public function add_media_button() {
        global $post;

        if($post->post_type !== 'product') {
            echo '<button id="elyssa-ai-media-open-modal" class="elyssa-ai-button" title="'.(!elyssa_is_image_enabled() ? esc_html('Please enable image services in the settings', 'elyssa-ai') : '').'" '.(!elyssa_is_image_enabled() ? 'disabled' : '').'><img src="' . esc_url(plugins_url('elyssa-ai/images/elyssa-icon-white.webp')) . '" alt="elyssa AI Logo" style="width:15px;height:15px;float:left;">' . esc_html__('elyssa AI Media', 'elyssa-ai') . '</button>';
        }
    }

    private function download_image_to_media_library($image_url, $post_id = 0, $description = '') {
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');

        elyssa_ai_log(__("Attempting to download image: ", 'elyssa-ai') . $image_url);

        $tmp = download_url($image_url);
        if (is_wp_error($tmp)) {
            elyssa_ai_log(__("Error downloading image: ", 'elyssa-ai') . $tmp->get_error_message());
            return false;
        }

        elyssa_ai_log(__("Image downloaded to temp file: ", 'elyssa-ai') . $tmp);

        $url_parts = wp_parse_url($image_url);
        $clean_path = pathinfo($url_parts['path']);
        $file_name = $clean_path['filename'] . '.' . $clean_path['extension'];

        $file_array = array(
            'name' => $file_name,
            'tmp_name' => $tmp
        );

        $file_type = wp_check_filetype($file_name);
        if (!$file_type['type']) {
            elyssa_ai_log(__("Invalid file type for image: ", 'elyssa-ai') . $file_name);
            wp_delete_file($tmp);
            return false;
        }

        elyssa_ai_log(__("Valid file type: ", 'elyssa-ai') . $file_type['type']);

        $attachment_id = media_handle_sideload($file_array, $post_id);

        if (is_wp_error($attachment_id)) {
            elyssa_ai_log(__("Error uploading image: ", 'elyssa-ai') . $attachment_id->get_error_message());
            wp_delete_file($tmp);
            return false;
        }

        elyssa_ai_log(__("Image uploaded successfully. Attachment ID: ", 'elyssa-ai') . $attachment_id);

        if (!empty($description)) {
            $update_result = wp_update_post(array(
                'ID' => $attachment_id,
                'post_excerpt' => $description
            ));
            if (is_wp_error($update_result)) {
                elyssa_ai_log(__("Error updating attachment description: ", 'elyssa-ai') . $update_result->get_error_message());
            } else {
                elyssa_ai_log(__("Attachment description updated successfully", 'elyssa-ai'));
            }
        }

        return $attachment_id;
    }
}

new ElyssaAIMedia();