<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

function elyssa_ai_register_settings()
{
    // Registriert eine neue Einstellung für den OpenAI API-Schlüssel
    register_setting('elyssa_ai_options_group', 'elyssa_ai_logging_enabled', [
        'type' => 'boolean',
        'description' => __('Activates or deactivates logging', 'elyssa-ai'),
        'default' => false,
        'sanitize_callback' => 'rest_sanitize_boolean',
    ]);
    register_setting('elyssa_ai_options_group', 'elyssa_ai_max_tokens', array(
        'type' => 'integer',
        'description' => __('Maximum token length', 'elyssa-ai'),
        'default' => 350, // Setze einen Standardwert, der sinnvoll ist
        'sanitize_callback' => 'absint', // Stellt sicher, dass der Wert eine positive ganze Zahl ist
    ));
    register_setting('elyssa_ai_options_group', 'elyssa_ai_timeout', array(
        'type' => 'integer',
        'description' => __('Timeout of the request to OpenAI', 'elyssa-ai'),
        'default' => 30, // Setze einen Standardwert, der sinnvoll ist
        'sanitize_callback' => 'absint', // Stellt sicher, dass der Wert eine positive ganze Zahl ist
    ));
    register_setting('elyssa_ai_options_group', 'elyssa_ai_text_service', [
        'type' => 'string',
        'description' => __('The AI service to be used', 'elyssa-ai'),
        'default' => '',
        'sanitize_callback' => 'sanitize_text_field',
    ]);

    register_setting('elyssa_ai_options_group', 'elyssa_ai_image_service', [
        'type' => 'string',
        'description' => __('The AI service to be used for images', 'elyssa-ai'),
        'default' => '',
        'sanitize_callback' => 'sanitize_text_field',
    ]);
    register_setting('elyssa_ai_options_group', 'elyssa_ai_text_api_key', [
        'type' => 'string',
        'description' => __('API key for text AI', 'elyssa-ai'),
        'sanitize_callback' => 'elyssa_ai_validate_api_key',
    ]);
    register_setting('elyssa_ai_options_group', 'elyssa_ai_image_api_key', [
        'type' => 'string',
        'description' => __('API key for image AI', 'elyssa-ai'),
        'sanitize_callback' => 'elyssa_ai_validate_api_key',
    ]);
    register_setting('elyssa_ai_options_group', 'elyssa_ai_comment_rating_enabled', [
        'type' => 'boolean',
        'description' => __('Activates or deactivates the AI rating of comments', 'elyssa-ai'),
        'default' => false,
        'sanitize_callback' => 'rest_sanitize_boolean',
    ]);
    
    register_setting('elyssa_ai_options_group', 'elyssa_ai_auto_update_enabled', [
        'type' => 'boolean',
        'description' => __('Enable automatic updates for Elyssa AI', 'elyssa-ai'),
        'default' => true,
        'sanitize_callback' => 'rest_sanitize_boolean',
    ]);

    register_setting('elyssa_ai_options_group', 'elyssa_ai_text_model', [
        'type' => 'string',
        'description' => __('The AI model to be used for text generation', 'elyssa-ai'),
        'default' => '',
        'sanitize_callback' => 'sanitize_text_field',
    ]);
    register_setting('elyssa_ai_options_group', 'elyssa_ai_image_model', [
        'type' => 'string',
        'description' => __('The AI model to be used for image generation', 'elyssa-ai'),
        'default' => '',
        'sanitize_callback' => 'sanitize_text_field',
    ]);
}

add_action('admin_init', 'elyssa_ai_register_settings');

/**
 * Enqueue settings page scripts and styles
 */
function elyssa_ai_settings_enqueue_scripts($hook) {
    if ($hook !== 'elyssa-ai_page_elyssa-ai-settings') {
        return;
    }

    // JavaScript für den Update-Check enqueuen
    wp_register_script(
        'elyssa-ai-settings-js',
        ELYSSA_AI_PLUGIN_URL . '/admin/assets/js/settings.js',
        array('jquery'),
        ELYSSA_AI_VERSION,
        true
    );
    
    // Lokalisierungsdaten für das Script
    wp_localize_script(
        'elyssa-ai-settings-js',
        'elyssaSettings',
        array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('elyssa_ai_check_update_nonce'),
            'checking_text' => __('Checking for updates...', 'elyssa-ai'),
            'up_to_date_text' => __('You have the latest version installed.', 'elyssa-ai'),
            'update_available_text' => __('Update available! Version', 'elyssa-ai'),
            'error_text' => __('Error checking for updates:', 'elyssa-ai')
        )
    );
    
    wp_enqueue_script('elyssa-ai-settings-js');
    
    // Styles registrieren und enqueuen
    wp_register_style(
        'elyssa-ai-settings-css',
        ELYSSA_AI_PLUGIN_URL . '/admin/assets/css/settings.css',
        array(),
        ELYSSA_AI_VERSION
    );
    
    wp_enqueue_style('elyssa-ai-settings-css');
}
add_action('admin_enqueue_scripts', 'elyssa_ai_settings_enqueue_scripts');

function elyssa_ai_settings_page()
{
    include ELYSSA_AI_PLUGIN_DIR . 'admin/html/header.php';
    $available_services = ElyssaAI_ServiceFactory::get_available_services();

    $text_service = get_option('elyssa_ai_text_service');
    $image_service = get_option('elyssa_ai_image_service');

    // No need to get available models anymore as we're using text fields

    ?>
    <div class="flex gap-5">
        <div class="w-3/5 mr-2">
            <?php
            // Check if settings were just updated
            if (isset($_GET['settings-updated']) && $_GET['settings-updated'] == 'true') {
                ?>
                <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 mb-4" role="alert">
                    <p><?php esc_html_e('Settings have been successfully updated.', 'elyssa-ai'); ?></p>
                </div>
                <?php
            }
            ?>
            <section class="bg-white p-6 shadow-md rounded-lg">
                <h2 class="text-xl font-semibold mb-4"><?php esc_html_e('Settings', 'elyssa-ai'); ?></h2>
                <div class="grid gap-8">
                    <form method="post" action="options.php">
                        <?php settings_fields('elyssa_ai_options_group'); ?>
                        <?php do_settings_sections('elyssa_ai_options_group'); ?>
                        <table class="form-table">
                            <tr valign="top">
                                <th scope="row"><?php esc_html_e('Activate comment rating', 'elyssa-ai'); ?></th>
                                <td><input type="checkbox" name="elyssa_ai_comment_rating_enabled"
                                           value="1" <?php checked(1, get_option('elyssa_ai_comment_rating_enabled'), true); ?> />
                                </td>
                            </tr>
                            <tr valign="top">
                                <th scope="row"><?php esc_html_e('Updates', 'elyssa-ai'); ?></th>
                                <td>
                                    <div style="margin-bottom:15px;">
                                        <input type="checkbox" name="elyssa_ai_auto_update_enabled"
                                               value="1" <?php checked(1, get_option('elyssa_ai_auto_update_enabled', true), true); ?> />
                                        <label for="elyssa_ai_auto_update_enabled"><?php esc_html_e('Enable automatic updates', 'elyssa-ai'); ?></label>
                                        <p class="description"><?php esc_html_e('Automatically install new versions of Elyssa AI when available.', 'elyssa-ai'); ?></p>
                                    </div>
                                    
                                    <div class="elyssa-ai-manual-update">
                                        <button type="button" id="elyssa-ai-check-update" class="button button-secondary">
                                            <?php esc_html_e('Check for updates now', 'elyssa-ai'); ?>
                                        </button>
                                        <span id="elyssa-ai-update-status" class="elyssa-ai-update-status" style="margin-left:10px;"><?php printf(esc_html__('Current version: %s', 'elyssa-ai'), ELYSSA_AI_VERSION); ?></span>
                                    </div>
                                </td>
                            </tr>
                            <tr valign="top">
                                <th scope="row"><?php esc_html_e('Activate logging', 'elyssa-ai'); ?></th>
                                <td><input type="checkbox" name="elyssa_ai_logging_enabled"
                                           value="1" <?php checked(1, get_option('elyssa_ai_logging_enabled'), true); ?> />
                                </td>
                            </tr>
                            <tr valign="top">
                                <th scope="row"><?php esc_html_e('Maximum token length', 'elyssa-ai'); ?></th>
                                <td><input type="number" name="elyssa_ai_max_tokens"
                                           value="<?php echo esc_attr(get_option('elyssa_ai_max_tokens', 350)); ?>"
                                           min="1"
                                           placeholder="<?php esc_attr_e('e.g. 350', 'elyssa-ai'); ?>"/>
                                    <p class="description"><?php esc_html_e('The maximum number of tokens for AI requests.', 'elyssa-ai'); ?></p>
                                </td>
                            </tr>
                            <tr valign="top">
                                <th scope="row"><?php esc_html_e('Timeout request (sec.)', 'elyssa-ai'); ?></th>
                                <td><input type="number" name="elyssa_ai_timeout"
                                           value="<?php echo esc_attr(get_option('elyssa_ai_timeout', 30)); ?>"
                                           min="1"
                                           placeholder="<?php esc_attr_e('e.g. 30', 'elyssa-ai'); ?>"/>
                                    <p class="description"><?php esc_html_e('The maximum waiting time for an AI response in seconds.', 'elyssa-ai'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th colspan="2">
                                    <hr/>
                                </th>
                            </tr>
                            <tr valign="top">
                                <th scope="row"><?php esc_html_e('Text-AI-Service', 'elyssa-ai'); ?></th>
                                <td>
                                    <select name="elyssa_ai_text_service" id="elyssa-ai-text-service">
                                        <option value="" <?php selected(get_option('elyssa_ai_text_service'), ''); ?>>
                                            <?php esc_html_e('-- Please select --', 'elyssa-ai'); ?>
                                        </option>
                                        <?php foreach ($available_services['text'] as $id => $class_name):
                                            $service_info = $class_name::get_service_info();
                                            $name = $service_info['name'];
                                            ?>
                                            <option value="<?php echo esc_attr($id); ?>" <?php selected(get_option('elyssa_ai_text_service'), $id); ?>>
                                                <?php echo esc_html($name); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <p class="description"><?php esc_html_e('Select the AI service for text generation.', 'elyssa-ai'); ?></p>
                                </td>
                            </tr>
                            <tr valign="top">
                                <th scope="row"><?php esc_html_e('Text AI Model', 'elyssa-ai'); ?></th>
                                <td>
                                    <input type="text" name="elyssa_ai_text_model" id="elyssa-ai-text-model"
                                           value="<?php echo esc_attr(get_option('elyssa_ai_text_model')); ?>"
                                           placeholder="<?php esc_attr_e('e.g. gpt-4-turbo', 'elyssa-ai'); ?>"/>
                                    <p class="description"><?php esc_html_e('Enter the model ID for text generation', 'elyssa-ai'); ?></p>
                                </td>
                            </tr>
                            <tr valign="top">
                                <th scope="row"><?php esc_html_e('Text-AI-API Key', 'elyssa-ai'); ?></th>
                                <td><input type="text" name="elyssa_ai_text_api_key"
                                           value="<?php echo esc_attr(elyssa_ai_get_api_key_option('elyssa_ai_text_api_key')); ?>"/>
                                    <p class="description"><?php esc_html_e('The API key for the selected text AI service.', 'elyssa-ai'); ?><br>
                                    <strong><?php esc_html_e('Note:', 'elyssa-ai'); ?></strong> <?php esc_html_e('API keys are stored encrypted in the database.', 'elyssa-ai'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th colspan="2">
                                    <hr/>
                                </th>
                            </tr>
                            <tr valign="top">
                                <th scope="row"><?php esc_html_e('Image-AI-Service', 'elyssa-ai'); ?></th>
                                <td>
                                    <select name="elyssa_ai_image_service" id="elyssa-ai-image-service">
                                        <option value="" <?php selected(get_option('elyssa_ai_image_service'), ''); ?>>
                                            <?php esc_html_e('-- Please select --', 'elyssa-ai'); ?>
                                        </option>
                                        <?php foreach ($available_services['image'] as $id => $class_name):
                                            $service_info = $class_name::get_service_info();
                                            $name = $service_info['name'];
                                            ?>
                                            <option value="<?php echo esc_attr($id); ?>" <?php selected(get_option('elyssa_ai_image_service'), $id); ?>>
                                                <?php echo esc_html($name); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <p class="description"><?php esc_html_e('Select the AI service for image generation.', 'elyssa-ai'); ?></p>
                                </td>
                            </tr>
                            <tr valign="top">
                                <th scope="row"><?php esc_html_e('Image AI Model', 'elyssa-ai'); ?></th>
                                <td>
                                    <input type="text" name="elyssa_ai_image_model" id="elyssa-ai-image-model"
                                           value="<?php echo esc_attr(get_option('elyssa_ai_image_model')); ?>"
                                           placeholder="<?php esc_attr_e('e.g. dall-e-3', 'elyssa-ai'); ?>"/>
                                    <p class="description"><?php esc_html_e('Enter the model ID for image generation', 'elyssa-ai'); ?></p>
                                </td>
                            </tr>
                            <tr valign="top">
                                <th scope="row"><?php esc_html_e('Image-AI API Key', 'elyssa-ai'); ?></th>
                                <td><input type="text" name="elyssa_ai_image_api_key"
                                           value="<?php echo esc_attr(elyssa_ai_get_api_key_option('elyssa_ai_image_api_key')); ?>"/>
                                    <p class="description"><?php esc_html_e('The API key for the selected image AI service.', 'elyssa-ai'); ?><br>
                                    <strong><?php esc_html_e('Note:', 'elyssa-ai'); ?></strong> <?php esc_html_e('API keys are stored encrypted in the database.', 'elyssa-ai'); ?></p>
                                </td>
                            </tr>
                        </table>
                        <?php submit_button(__('Save changes', 'elyssa-ai')); ?>
                    </form>
                </div>
            </section>
        </div>

        <div class="w-2/5 space-y-10">
            <section class="bg-white p-6 shadow-md rounded-lg">
                <h2 class="text-xl font-semibold mb-4"><?php esc_html_e('Update Information', 'elyssa-ai'); ?></h2>
                <p><?php esc_html_e('Current Version:', 'elyssa-ai'); ?> <strong><?php echo ELYSSA_AI_VERSION; ?></strong></p>
                <p><?php esc_html_e('Update Server:', 'elyssa-ai'); ?> <code>https://get.codefabrica.net/wp/elyssa-ai</code></p>
                
                <?php if (current_user_can('manage_options') && get_option('elyssa_ai_logging_enabled')): ?>
                <div class="elyssa-ai-debug-box" style="margin-top: 15px; padding: 10px; background: #f5f5f5; border: 1px solid #ddd;">
                    <h3 style="margin-top: 0;"><?php esc_html_e('Developer Debug Info', 'elyssa-ai'); ?></h3>
                    <p><?php esc_html_e('Last check result:', 'elyssa-ai'); ?> <span id="elyssa-ai-debug-info"></span></p>
                    <button type="button" id="elyssa-ai-test-connection" class="button button-secondary">
                        <?php esc_html_e('Test Update Server Connection', 'elyssa-ai'); ?>
                    </button>
                    <div id="elyssa-ai-transient-info" style="margin-top: 10px;"></div>
                </div>
                
                <div class="elyssa-ai-debug-actions" style="margin-top: 15px; padding: 10px; background: #f8f8f8; border: 1px solid #ddd;">
                    <h3 style="margin-top: 0;"><?php esc_html_e('Debug Actions', 'elyssa-ai'); ?></h3>
                    <p>
                        <button type="button" id="elyssa-ai-reset-transients" class="button button-secondary">
                            <?php esc_html_e('Reset Update Transients', 'elyssa-ai'); ?>
                        </button>
                        <span style="margin-left: 10px; color: #666;"><?php esc_html_e('Use this if WordPress is not detecting updates.', 'elyssa-ai'); ?></span>
                    </p>
                </div>
                <?php endif; ?>
            </section>
                

            <section class="bg-white p-6 shadow-md rounded-lg">
                <h2 class="text-xl font-semibold mb-4"><?php esc_html_e('Support', 'elyssa-ai'); ?></h2>
                <p><?php esc_html_e('Any ideas or bugs? Please contact us by email: support@codefabrica.net.', 'elyssa-ai'); ?></p>
                <p><?php esc_html_e('Please note that we only provide limited support for free products.', 'elyssa-ai') ?></p>
                <p class="mt-4">
                    <a href="https://elyssaai.com" target="_blank"
                       class="elyssa-ai-button">
                        <?php esc_html_e('Visit our website', 'elyssa-ai'); ?>
                    </a>
                </p>
            </section>
            <section class="bg-white p-6 shadow-md rounded-lg">
                <h2 class="text-xl font-semibold mb-4"><?php esc_html_e('Documentation', 'elyssa-ai'); ?></h2>
                <p><?php esc_html_e('You\'ll find the latest documentation for elyssa AI on our documentation platform:', 'elyssa-ai') ?></p>
                <p class="mt-4">
                    <a href="https://elyssaai.com/documentation" target="_blank"
                       class="elyssa-ai-button">
                        <?php esc_html_e('Visit documentation', 'elyssa-ai'); ?>
                    </a>
                </p>
            </section>
        </div>
    </div>
    <?php
    include ELYSSA_AI_PLUGIN_DIR . 'admin/html/footer.php';
}


function elyssa_ai_validate_api_key($input)
{
    if (!is_string($input)) {
        add_settings_error('elyssa_ai_text_api_key', 'elyssa_ai_api_key_error', __('Invalid API key.', 'elyssa-ai'));
        return '';
    }
    
    // Wenn der Input leer ist, geben wir einen leeren String zurück
    if (empty($input)) {
        return '';
    }
    
    // Verschlüssle den API-Schlüssel
    return elyssa_ai_encrypt_api_key($input);
}


// Ajax-Funktion für Modelle wurde entfernt, da Modelle jetzt manuell eingegeben werden

function elyssa_ai_validate_settings($input) {
    if (empty($input['elyssa_ai_text_service'])) {
        add_settings_error('elyssa_ai_text_service', 'elyssa_ai_text_service_error', __('Please select a Text AI service.', 'elyssa-ai'));
    }
    if (empty($input['elyssa_ai_image_service'])) {
        add_settings_error('elyssa_ai_image_service', 'elyssa_ai_image_service_error', __('Please select an Image AI service.', 'elyssa-ai'));
    }
    return $input;
}
add_filter('pre_update_option_elyssa_ai_options_group', 'elyssa_ai_validate_settings');

/**
 * AJAX-Handler für die Prüfung der WordPress-Transients
 */
function elyssa_ai_check_transients_ajax() {
    // Überprüfe die Nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'elyssa_ai_check_update_nonce')) {
        wp_send_json_error(array('message' => __('Security check failed.', 'elyssa-ai')));
    }
    
    // Überprüfe Berechtigungen
    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => __('You do not have permission to perform this action.', 'elyssa-ai')));
    }
    
    // Plugin-Informationen abrufen
    $plugin_basename = plugin_basename(ELYSSA_AI_PLUGIN_FILE);
    $plugin_slug = dirname($plugin_basename);
    
    // Transients abrufen
    $update_plugins = get_site_transient('update_plugins');
    $plugin_info = get_site_transient('update_plugins');
    
    // Für spezifisches Plugin
    $elyssa_transient = get_transient('elyssa_ai_update_check');
    $elyssa_plugin_info = get_transient('elyssa_ai_plugin_info');
    
    // Direkte Prüfung durch WP 
    $args = array(
        'slug' => $plugin_slug,
        'version' => ELYSSA_AI_VERSION,
    );
    
    // Force ein Update-Check
    require_once ABSPATH . 'wp-admin/includes/plugin.php';
    wp_clean_plugins_cache(true);
    delete_site_transient('update_plugins');
    wp_update_plugins();
    $update_plugins_new = get_site_transient('update_plugins');
    
    // Rückgabe
    $result = array(
        'plugin_basename' => $plugin_basename,
        'plugin_slug' => $plugin_slug,
        'version' => ELYSSA_AI_VERSION,
        'update_plugins_transient' => is_object($update_plugins) ? (array)$update_plugins : 'Not an object', 
        'update_plugins_response' => is_object($update_plugins) && isset($update_plugins->response) ? (array)$update_plugins->response : 'No response',
        'update_plugins_checked' => is_object($update_plugins) && isset($update_plugins->checked) ? (array)$update_plugins->checked : 'No checked',
        'elyssa_transient' => $elyssa_transient,
        'elyssa_plugin_info' => $elyssa_plugin_info,
        'after_force_check' => is_object($update_plugins_new) && isset($update_plugins_new->response) ? array_keys((array)$update_plugins_new->response) : 'No response after force check',
        'after_checked' => is_object($update_plugins_new) && isset($update_plugins_new->checked) ? (array)$update_plugins_new->checked : 'No checked after force check',
    );
    
    // Loggen, wenn aktiviert
    if (get_option('elyssa_ai_logging_enabled')) {
        elyssa_ai_log("Transient check completed");
    }
    
    wp_send_json_success($result);
}
add_action('wp_ajax_elyssa_ai_check_transients', 'elyssa_ai_check_transients_ajax');
