document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('elyssa-ai-bulk-form');
    const generateTitlesButton = document.getElementById('elyssa-ai-generate-titles');
    const titlesContainer = document.getElementById('elyssa-ai-titles-container');
    const titlesList = document.getElementById('elyssa-ai-titles-list');
    const actionBox = document.getElementById('elyssa-ai-action-box');
    const generateContentButton = document.getElementById('elyssa-ai-generate-content');
    const addManualTitleButton = document.getElementById('elyssa-ai-add-manual-title');
    const manualTitleInput = document.getElementById('elyssa-ai-manual-title-input');
    const contentTypeSelect = document.getElementById('elyssa-ai-content-type');

    generateTitlesButton.addEventListener('click', function(e) {
        e.preventDefault();
        generateTitles();
    });

    addManualTitleButton.addEventListener('click', function() {
        addManualTitle();
    });

    generateContentButton.addEventListener('click', function() {
        generateContent();
    });

    function showLoadingIndicator() {
        if (!document.getElementById('elyssaLoadingIndicator')) {
            const indicator = document.createElement('div');
            indicator.id = 'elyssaLoadingIndicator';
            indicator.textContent = 'Loading...';

            Object.assign(indicator.style, {
                position: 'fixed',
                top: '0',
                left: '0',
                width: '100%',
                height: '100%',
                backgroundColor: 'rgba(0, 0, 0, 0.5)',
                color: 'white',
                fontSize: '24px',
                display: 'flex',
                justifyContent: 'center',
                alignItems: 'center',
                zIndex: '9999'
            });

            document.body.appendChild(indicator);
        } else {
            document.getElementById('elyssaLoadingIndicator').style.display = 'flex';
        }
    }

    function hideLoadingIndicator() {
        const indicator = document.getElementById('elyssaLoadingIndicator');
        if (indicator) {
            indicator.style.display = 'none';
        }
    }

    function generateTitles() {
        const formData = new FormData(form);
        formData.append('action', 'elyssa_ai_generate_titles');
        formData.append('nonce', elyssaAiBulk.nonce);

        showLoadingIndicator();

        fetch(elyssaAiBulk.ajax_url, {
            method: 'POST',
            credentials: 'same-origin',
            body: formData
        })
            .then(response => response.json())
            .then(data => {
                hideLoadingIndicator();
                if (data.success) {
                    displayTitles(data.data.headlines);
                } else {
                    console.error(elyssaAiBulk.error_generating_titles, data.data);
                    alert(elyssaAiBulk.error_generating_titles + ' ' + data.data);
                }
            })
            .catch(error => {
                hideLoadingIndicator();
                console.error('Error:', error);
                alert(elyssaAiBulk.error_generating_titles + ' ' + error.message);
            });
    }

    function displayTitles(titles) {
        titlesList.innerHTML = '';
        const originalKeywords = document.getElementById('elyssa-ai-bulk-keywords').value;
        titles.forEach((title, index) => {
            const row = titlesList.insertRow();
            row.innerHTML = `
    <td class="w-1/12">
        <input type="checkbox" id="title-${index}" name="selected_titles[]" value="${title}">
    </td>
    <td class="w-5/12">
        <label for="title-${index}" class="font-semibold">${title}</label>
    </td>
    <td class="w-4/12">
        <input type="text" name="keywords[]" value="${originalKeywords}" placeholder="${elyssaAiBulk.keywords_placeholder}" class="w-full">
    </td>
    <td class="w-2/12 entry_status font-semibold"></td>
`;
        });
        titlesContainer.style.display = 'block';
        actionBox.style.display = 'block';
        contentTypeSelect.disabled = true;
        generateContentButton.disabled = true;
        contentTypeSelect.value = '';
    }

    titlesList.addEventListener('change', function(e) {
        if (e.target.type === 'checkbox') {
            updateGenerateButtonState();
        }
    });

    function addManualTitle() {
        const title = manualTitleInput.value.trim();
        if (title) {
            const index = titlesList.rows.length;
            const newRow = titlesList.insertRow();
            newRow.innerHTML = `
            <td>
                <input type="checkbox" id="title-${index}" name="selected_titles[]" value="${title}">
            </td>
            <td>
                <label for="title-${index}">${title}</label>
            </td>
            <td>
                <input type="text" name="keywords[]" placeholder="${elyssaAiBulk.keywords_placeholder}" class="w-full">
            </td>
            <td class="w-2/12 entry_status font-semibold"></td>
        `;
            manualTitleInput.value = '';
            updateGenerateButtonState();
        }
    }

    function updateGenerateButtonState() {
        const checkedTitles = titlesList.querySelectorAll('input[type="checkbox"]:checked');
        const isAnyTitleChecked = checkedTitles.length > 0;

        generateContentButton.disabled = !isAnyTitleChecked;
        contentTypeSelect.disabled = !isAnyTitleChecked;

        if (isAnyTitleChecked) {
            contentTypeSelect.value = '';
        } else {
            contentTypeSelect.value = '';
        }
    }

    function generateContent() {
        if (contentTypeSelect.value === '') {
            alert(elyssaAiBulk.please_select_content_type);
            return;
        }

        const selectedTitles = Array.from(titlesList.querySelectorAll('input[type="checkbox"]:checked'));
        const contentType = contentTypeSelect.value;

        if (selectedTitles.length === 0) {
            alert(elyssaAiBulk.no_titles_selected);
            return;
        }

        showLoadingIndicator();

        selectedTitles.forEach(checkbox => {
            const row = checkbox.closest('tr');
            if (!row) return;

            let statusCell = row.querySelector('.entry_status');
            if (!statusCell) {
                statusCell = document.createElement('td');
                statusCell.className = 'entry_status';
                row.appendChild(statusCell);
            }
            statusCell.textContent = elyssaAiBulk.generating;
            statusCell.style.color = 'blue';
        });

        const contentPromises = selectedTitles.map(checkbox => {
            const row = checkbox.closest('tr');
            if (!row) return Promise.resolve(null);

            const title = checkbox.value;
            const keywordsInput = row.querySelector('input[name="keywords[]"]');
            const keywords = keywordsInput ? keywordsInput.value : '';

            const formData = new FormData();
            formData.append('action', 'elyssa_ai_generate_bulk_content');
            formData.append('nonce', elyssaAiBulk.nonce);
            formData.append('title', title);
            formData.append('keywords', keywords);
            formData.append('content_type', contentType);

            return fetch(elyssaAiBulk.ajax_url, {
                method: 'POST',
                credentials: 'same-origin',
                body: formData
            })
                .then(response => response.json())
                .then(data => {
                    const statusCell = row.querySelector('.entry_status');
                    if (!statusCell) return null;

                    if (data.success) {
                        statusCell.textContent = elyssaAiBulk.success;
                        statusCell.style.color = 'green';
                        const titleLabel = row.querySelector('label');
                        if (titleLabel && data.data.title !== title) {
                            titleLabel.textContent = data.data.title;
                        }
                        return data.data;
                    } else {
                        statusCell.textContent = elyssaAiBulk.error + ' ' + data.data;
                        statusCell.style.color = 'red';
                        throw new Error(data.data);
                    }
                })
                .catch(error => {
                    const statusCell = row.querySelector('.entry_status');
                    if (statusCell) {
                        statusCell.textContent = elyssaAiBulk.error + ' ' + error.message;
                        statusCell.style.color = 'red';
                    }
                    console.error(elyssaAiBulk.error_generating_content, error);
                    return null;
                });
        });

        Promise.all(contentPromises)
            .then(results => {
                hideLoadingIndicator();
                const successfulResults = results.filter(result => result !== null);
                if (successfulResults.length > 0) {
                    displayResults(successfulResults);
                } else {
                    alert(elyssaAiBulk.no_content_generated);
                }
            });
    }

    function displayResults(results) {
        let message = elyssaAiBulk.generated_items.replace('%d', results.length) + '\n\n';
        results.forEach(result => {
            message += `- ${result.title} (ID: ${result.id})\n`;
        });
        message += '\n' + elyssaAiBulk.find_in_drafts;
        alert(message);
    }
});