<?php
if (!defined('ABSPATH')) {
    exit;
}

class ElyssaAIBulk extends ElyssaAIBaseService
{
    private static $instance;

    public static function get_instance()
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        add_action('admin_enqueue_scripts', array($this, 'enqueue_bulk_scripts'));
        add_action('wp_ajax_elyssa_ai_generate_titles', array($this, 'generate_titles'));
        add_action('wp_ajax_elyssa_ai_generate_bulk_content', array($this, 'generate_bulk_content'));
    }

    public function bulk_page()
    {
        if (!current_user_can('edit_posts')) {
            wp_die(esc_html('You do not have sufficient permissions to access this page.', 'elyssa-ai'));
        }
        include ELYSSA_AI_ROOT_PATH . 'admin/html/header.php';
        ?>
        <div class="w-full space-y-8">
            <section class="bg-white p-6 shadow-md rounded-lg">
                <h2 class="text-xl font-semibold mb-4"><?php esc_html_e('Bulk Generation', 'elyssa-ai'); ?></h2>
                <p class="mb-3 font-normal"><?php esc_html_e('What story would you like to tell today?', 'elyssa-ai') ?></p>
                <div class="gap-8">
                    <form id="elyssa-ai-bulk-form">
                        <table class="form-table">
                            <tr valign="top">
                                <th scope="row"> <label for="elyssa-ai-bulk-keywords"><?php esc_html_e('Keywords (separated by comma)', 'elyssa-ai'); ?></label> </th>
                                <td><input type="text" id="elyssa-ai-bulk-keywords" name="keywords" required /></td>
                            </tr>
                            <tr valign="top">
                                <th scope="row"> <label for="elyssa-ai-bulk-count"><?php esc_html_e('Number of Articles', 'elyssa-ai'); ?></label></th>
                                <td><input type="number" id="elyssa-ai-bulk-count" name="count" min="1" max="25" value="10" required /></td>
                            </tr>
                        </table>
                        <button type="button"
                                id="elyssa-ai-generate-titles"
                                <?php if(!elyssa_is_text_enabled()): ?>title="<?php esc_html_e('Please enable text services in the settings', 'elyssa-ai'); ?>"<?php endif;?>
                                class="elyssa-ai-button"
                                <?php if(!elyssa_is_text_enabled()): ?>disabled<?php endif;?>
                        ><?php esc_html_e('Generate Titles', 'elyssa-ai'); ?></button>
                    </form>
                    <div id="elyssa-ai-titles-container" class="mt-10" style="display:none;">
                        <h2 class="font-semibold text-lg"><?php esc_html_e('Generated Titles', 'elyssa-ai'); ?></h2>
                        <p class="my-3"><?php esc_html_e('Please select the desired titles you want to create content for.', 'elyssa-ai') ?></p>
                        <table class="form-table" id="elyssa-ai-titles-list"></table>
                        <div id="elyssa-ai-manual-title">
                            <input type="text" id="elyssa-ai-manual-title-input"
                                   placeholder="<?php esc_html_e('Add manual title', 'elyssa-ai'); ?>">
                            <button id="elyssa-ai-add-manual-title"
                                    class="elyssa-ai-button"><?php esc_html_e('Add Title', 'elyssa-ai'); ?></button>
                        </div>
                    </div>
                    <div id="elyssa-ai-action-box" class="mt-10" style="display:none;">
                        <h2 class="font-semibold text-lg"><?php esc_html_e('Create Content', 'elyssa-ai'); ?></h2>
                        <p class="my-3"><?php esc_html_e('Page/Post will be generated in Draft Mode.', 'elyssa-ai') ?></p>
                        <select id="elyssa-ai-content-type">
                            <option value=""><?php esc_html_e('Please select', 'elyssa-ai'); ?></option>
                            <option value="post"><?php esc_html_e('Posts', 'elyssa-ai'); ?></option>
                            <option value="page"><?php esc_html_e('Pages', 'elyssa-ai'); ?></option>
                        </select>
                        <button id="elyssa-ai-generate-content" class="elyssa-ai-button"
                                disabled><?php esc_html_e('Generate', 'elyssa-ai'); ?></button>
                    </div>
                </div>
            </section>
        </div>
        <div id="elyssa-ai-loading" style="display:none;">
            <div class="elyssa-ai-loading-content">
                <p><?php esc_html_e('Loading...', 'elyssa-ai'); ?></p>
            </div>
        </div>
        <?php
        include ELYSSA_AI_ROOT_PATH . 'admin/html/footer.php';
    }

    public function enqueue_bulk_scripts($hook)
    {
        if ('elyssa-ai_page_elyssa-ai-bulk' !== $hook) {
            return;
        }

        wp_enqueue_script('elyssa-ai-bulk', plugin_dir_url(__FILE__) . 'assets/elyssa-ai-bulk.js?' . time(), array('jquery'), ELYSSA_AI_VERSION, true);
        wp_localize_script('elyssa-ai-bulk', 'elyssaAiBulk', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('elyssa_ai_bulk_nonce'),
            'please_select_content_type' => __('Please select a content type (Post or Page) before generating.', 'elyssa-ai'),
            'error_generating_titles' => __('Error generating titles:', 'elyssa-ai'),
            'error_getting_title_suggestions' => __('Error getting title suggestions:', 'elyssa-ai'),
            'error_generating_content' => __('Error generating content:', 'elyssa-ai'),
            /* translators: number of items to translate */
            'generated_items' => __('Generated %d item(s):', 'elyssa-ai'),
            'find_in_drafts' => __('You can find these items in your drafts.', 'elyssa-ai'),
            'invalid_input' => __('Please provide keywords and a valid number of titles to generate.', 'elyssa-ai'),
            'no_titles_selected' => __('Please select at least one title to generate content.', 'elyssa-ai'),
            'no_content_generated' => __('No content was successfully generated. Please try again.', 'elyssa-ai'),
            'generating' => __('Generating...', 'elyssa-ai'),
            'success' => __('Success', 'elyssa-ai'),
            'error' => __('Error:', 'elyssa-ai'),
            'keywords_placeholder' => __('Keywords (comma separated)', 'elyssa-ai'),
        ));
    }

    public function generate_titles()
    {
        check_ajax_referer('elyssa_ai_bulk_nonce', 'nonce');

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(__('You do not have sufficient permissions to perform this action.', 'elyssa-ai'));
            return;
        }

        $keywords = sanitize_text_field(wp_unslash($_POST['keywords'] ?? ''));
        $count = intval($_POST['count'] ?? 0);

        if (empty($keywords) || $count <= 0) {
            wp_send_json_error(__('Invalid input. Please provide keywords and a valid count.', 'elyssa-ai'));
            return;
        }

        $init_result = $this->init_text_service();
        if (is_wp_error($init_result)) {
            wp_send_json_error($init_result->get_error_message());
            return;
        }

        if (!$this->text_service) {
            wp_send_json_error(__('Text AI Service not initialized. Please select a service in the settings.', 'elyssa-ai'));
            return;
        }

        $system_prompt = ServicePrompts::get_system('generate_titles');
        $user_prompt = sprintf(ServicePrompts::get_user('generate_titles'), $count, $keywords);

        try {
            $response = $this->text_service->generate_text($user_prompt, [], $system_prompt);

            // Parse the response
            $lines = explode("\n", $response);
            $titles = [];
            foreach ($lines as $line) {
                if (preg_match('/^\d+\.\s*(.+)$/', $line, $matches)) {
                    $titles[] = trim($matches[1]);
                }
            }

            if (count($titles) < $count) {
                wp_send_json_error(__('Insufficient titles generated. Please try again.', 'elyssa-ai'));
                return;
            }

            wp_send_json_success(['headlines' => $titles]);
        } catch (Exception $e) {
            wp_send_json_error(__('Error generating titles:', 'elyssa-ai') . ' ' . $e->getMessage());
        }
    }

    public function generate_bulk_content()
    {
        check_ajax_referer('elyssa_ai_bulk_nonce', 'nonce');

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(__('You do not have sufficient permissions to perform this action.', 'elyssa-ai'));
            return;
        }

        $title = sanitize_text_field(wp_unslash($_POST['title'] ??  '' ));
        $keywords = sanitize_text_field(wp_unslash($_POST['keywords'] ??  ''));
        $content_type = sanitize_text_field(wp_unslash($_POST['content_type'] ??  ''));

        $init_result = $this->init_text_service();
        if (is_wp_error($init_result)) {
            wp_send_json_error($init_result->get_error_message());
            return;
        }

        if (!$this->text_service) {
            wp_send_json_error(__('Text AI Service not initialized. Please select a service in the settings.', 'elyssa-ai'));
            return;
        }

        $system_prompt = ServicePrompts::get_system('generate_bulk_content');
        $user_prompt = ServicePrompts::get_user('generate_bulk_content');

        $prompt = sprintf(
            $user_prompt,
            $title,
            $keywords,
            $content_type === 'post' ? 'blog post' : 'page'
        );

        try {
            $response = $this->text_service->generate_text($prompt, [], $system_prompt);
            $ai_data = parse_structured_content($response);

            if (!isset($ai_data['title']) || !isset($ai_data['content']) || !isset($ai_data['keywords'])) {
                wp_send_json_error(__('Incomplete response from AI service. Please try again.', 'elyssa-ai'));
                return;
            }

            $post_id = wp_insert_post(array(
                'post_title' => $ai_data['title'],
                'post_content' => $ai_data['content'],
                'post_status' => 'draft',
                'post_type' => $content_type,
            ));

            if (is_wp_error($post_id)) {
                wp_send_json_error($post_id->get_error_message());
                return;
            }

            wp_set_post_tags($post_id, $ai_data['keywords']);

            wp_send_json_success(array(
                'id' => $post_id,
                'title' => $ai_data['title'],
                'edit_url' => get_edit_post_link($post_id, 'raw'),
            ));
        } catch (Exception $e) {
            wp_send_json_error(__('Error generating content:', 'elyssa-ai') . ' ' . $e->getMessage());
        }
    }
}

// Instantiate the class
ElyssaAIBulk::get_instance();