(function() {
    let chatModal;
    let chatMessages;
    let chatInput;
    let chatSendButton;
    let chatProvider;
    let chatModel;
    let chatContext = '';
    let isUnsavedChanges = false;

    function createChatModal() {
        // Erstelle das Chat-Modal-Element
        const container = document.getElementById('elyssa-ai-chat-container');
        
        // Modal-Struktur erstellen
        const modalHtml = `
            <div id="elyssa-ai-chat-modal" class="elyssa-ai-chat-modal">
                <div class="elyssa-ai-chat-modal-content" style="border-radius:9px;">
                    <div id="elyssa-ai-chat-controls">
                        <button id="elyssa-ai-save-chat" title="${wp.i18n.__('Save Chat', 'elyssa-ai')}">
                            <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M19 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h11l5 5v11a2 2 0 0 1-2 2z"></path>
                                <polyline points="17 21 17 13 7 13 7 21"></polyline>
                                <polyline points="7 3 7 8 15 8"></polyline>
                            </svg>
                        </button>
                        <button id="elyssa-ai-delete-chat" title="${wp.i18n.__('Delete Chat', 'elyssa-ai')}">
                            <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <polyline points="3 6 5 6 21 6"></polyline>
                                <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path>
                                <line x1="10" y1="11" x2="10" y2="17"></line>
                                <line x1="14" y1="11" x2="14" y2="17"></line>
                            </svg>
                        </button>
                        <span class="elyssa-ai-chat-close" title="${wp.i18n.__('Close Chat', 'elyssa-ai')}">
                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" width="24" height="24" stroke-width="3" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M6 18 18 6M6 6l12 12" />
                            </svg>
                        </span>
                    </div>
                    <div style="width:100%;">
                        <img src="${elyssaAiChatVars.logoUrl}" alt="elyssa AI Logo" style="height:60px;float:left;">
                    </div>
                    <div id="elyssa-ai-chat-messages"></div>
                    <div id="elyssa-ai-chat-info">
                        <span id="elyssa-ai-chat-provider"></span>
                        <span id="elyssa-ai-chat-model"></span>
                    </div>
                    <div id="elyssa-ai-chat-input-area">
                        <textarea id="elyssa-ai-chat-input" rows="3" style="padding:3px 5px;"></textarea>
                        <button
                            id="elyssa-ai-chat-send"
                            ${!elyssaAiChatVars.isTextEnabled ? `title="${elyssaAiChatVars.serviceDisabledText}"` : ''}
                            ${!elyssaAiChatVars.isTextEnabled ? 'disabled' : ''}
                        >${elyssaAiChatVars.sendText}</button>
                    </div>
                </div>
            </div>
        `;
        
        // HTML in den Container einfügen
        container.innerHTML = modalHtml;
    }

    function initChat() {
        // Erstelle das Chat-Modal, falls es noch nicht existiert
        createChatModal();
        
        chatModal = document.getElementById('elyssa-ai-chat-modal');
        chatMessages = document.getElementById('elyssa-ai-chat-messages');
        chatInput = document.getElementById('elyssa-ai-chat-input');
        chatSendButton = document.getElementById('elyssa-ai-chat-send');
        chatProvider = document.getElementById('elyssa-ai-chat-provider');
        chatModel = document.getElementById('elyssa-ai-chat-model');

        document.querySelector('.elyssa-ai-chat-close').onclick = handleCloseChat;
        document.getElementById('elyssa-ai-save-chat').onclick = saveChatHistory;
        document.getElementById('elyssa-ai-delete-chat').onclick = deleteChatHistory;
        chatSendButton.onclick = sendMessage;
        chatInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                sendMessage();
            }
        });

        updateProviderInfo(elyssaAiChatVars.provider, elyssaAiChatVars.model);
        loadChatHistory();
    }

    function openChat() {
        chatModal.style.display = 'block';
        chatInput.focus();
    }

    function handleCloseChat() {
        if (isUnsavedChanges) {
            const shouldSave = confirm(wp.i18n.__('Do you want to save the chat before closing?', 'elyssa-ai'));
            if (shouldSave) {
                saveChatHistory().then(() => {
                    closeChat();
                });
            } else {
                closeChat();
            }
        } else {
            closeChat();
        }
    }

    function closeChat() {
        chatModal.style.display = 'none';
        isUnsavedChanges = false;
    }

    function sendMessage() {
        const message = chatInput.value.trim();
        if (message) {
            addMessage('user', message);
            chatInput.value = '';
            isUnsavedChanges = true;

            const formData = new FormData();
            formData.append('action', 'elyssa_ai_chat_message');
            formData.append('nonce', elyssaAiChatVars.nonce);
            formData.append('message', message);
            formData.append('context', chatContext);

            addMessage('assistant', wp.i18n.__('Thinking...', 'elyssa-ai'), 'loading');

            fetch(elyssaAiChatVars.ajax_url, {
                method: 'POST',
                credentials: 'same-origin',
                body: formData
            })
                .then(response => response.json())
                .then(data => {
                    removeLoadingMessage();
                    if (data.success) {
                        addMessage('assistant', data.data.message);
                        updateProviderInfo(data.data.provider, data.data.model);
                    } else {
                        addMessage('error', wp.i18n.__('Error:', 'elyssa-ai') + ' ' + data.data.message);
                    }
                })
                .catch(error => {
                    removeLoadingMessage();
                    addMessage('error', wp.i18n.__('An error occurred while sending the message.', 'elyssa-ai'));
                    console.error('Error:', error);
                });
        }
    }

    function addMessage(sender, message, className = '') {
        const messageElement = document.createElement('div');
        messageElement.classList.add('chat-message', sender);
        if (className) {
            messageElement.classList.add(className);
        }
        messageElement.textContent = message;
        chatMessages.appendChild(messageElement);
        chatMessages.scrollTop = chatMessages.scrollHeight;

        // Update context
        chatContext += `${sender === 'user' ? 'Human' : 'Assistant'}: ${message}\n`;
        isUnsavedChanges = true;
    }

    function removeLoadingMessage() {
        const loadingMessage = chatMessages.querySelector('.loading');
        if (loadingMessage) {
            loadingMessage.remove();
        }
    }

    function updateProviderInfo(provider, model) {
        chatProvider.textContent = wp.i18n.__('Provider:', 'elyssa-ai') + ' ' + provider;
        chatModel.textContent = wp.i18n.__('Model:', 'elyssa-ai') + ' ' + model;
    }

    function saveChatHistory() {
        const formData = new FormData();
        formData.append('action', 'elyssa_ai_save_chat');
        formData.append('nonce', elyssaAiChatVars.nonce);
        formData.append('chat_history', chatContext);

        return fetch(elyssaAiChatVars.ajax_url, {
            method: 'POST',
            credentials: 'same-origin',
            body: formData
        })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert(wp.i18n.__('Chat history saved successfully.', 'elyssa-ai'));
                    isUnsavedChanges = false;
                } else {
                    alert(wp.i18n.__('Error saving chat history:', 'elyssa-ai') + ' ' + data.data.message);
                }
            })
            .catch(error => {
                alert(wp.i18n.__('An error occurred while saving the chat history.', 'elyssa-ai'));
                console.error('Error:', error);
            });
    }

    function loadChatHistory() {
        const formData = new FormData();
        formData.append('action', 'elyssa_ai_load_chat');
        formData.append('nonce', elyssaAiChatVars.nonce);

        fetch(elyssaAiChatVars.ajax_url, {
            method: 'POST',
            credentials: 'same-origin',
            body: formData
        })
            .then(response => response.json())
            .then(data => {
                if (data.success && data.data.chat_history) {
                    chatContext = data.data.chat_history;
                    const messages = chatContext.split('\n');
                    messages.forEach(message => {
                        if (message.startsWith('Human: ')) {
                            addMessage('user', message.substring(7));
                        } else if (message.startsWith('Assistant: ')) {
                            addMessage('assistant', message.substring(11));
                        }
                    });
                    isUnsavedChanges = false;
                }
            })
            .catch(error => {
                console.error(wp.i18n.__('An error occurred while loading the chat history:', 'elyssa-ai'), error);
            });
    }

    function deleteChatHistory() {
        if (confirm(wp.i18n.__('Are you sure you want to delete the chat history?', 'elyssa-ai'))) {
            const formData = new FormData();
            formData.append('action', 'elyssa_ai_delete_chat');
            formData.append('nonce', elyssaAiChatVars.nonce);

            fetch(elyssaAiChatVars.ajax_url, {
                method: 'POST',
                credentials: 'same-origin',
                body: formData
            })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        chatContext = '';
                        chatMessages.innerHTML = '';
                        alert(wp.i18n.__('Chat history deleted successfully.', 'elyssa-ai'));
                        isUnsavedChanges = false;
                    } else {
                        alert(wp.i18n.__('Error deleting chat history:', 'elyssa-ai') + ' ' + data.data.message);
                    }
                })
                .catch(error => {
                    alert(wp.i18n.__('An error occurred while deleting the chat history.', 'elyssa-ai'));
                    console.error('Error:', error);
                });
        }
    }

    document.addEventListener('DOMContentLoaded', initChat);

    window.elyssaAIOpenChat = openChat;
})();