<?php
if (!defined('ABSPATH')) {
    exit;
}

class ElyssaAIComments extends ElyssaAIBaseService
{
    public function __construct()
    {
        add_action('plugins_loaded', array($this, 'init'));
        remove_action('wp_insert_comment', array($this, 'auto_approve_comments'));
    }

    public function init()
    {
        if (get_option('elyssa_ai_comment_rating_enabled', false)) {
            // WordPress Standard-Kommentar-Moderation deaktivieren
            remove_action('pre_comment_approved', 'wp_approve_comment', 10);
            remove_action('comment_post', 'wp_new_comment_notify_postauthor');
            remove_action('comment_post', 'wp_new_comment_notify_moderator');
            
            // Alle Kommentare standardmäßig auf "hold" setzen
            add_filter('pre_comment_approved', function() { return 0; }, 99);

            add_filter('manage_edit-comments_columns', array($this, 'add_comment_column'));
            add_action('manage_comments_custom_column', array($this, 'comment_column_content'), 10, 2);
            add_filter('bulk_actions-edit-comments', array($this, 'add_bulk_actions'));
            add_filter('handle_bulk_actions-edit-comments', array($this, 'handle_bulk_actions'), 10, 3);
            add_action('admin_notices', array($this, 'bulk_action_admin_notice'));
            add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
            add_action('wp_ajax_elyssa_rate_comments', array($this, 'ajax_rate_comments'));
            add_action('comment_post', array($this, 'rate_new_comment'), 10, 2);

            add_filter('manage_edit-comments_columns', array($this, 'remove_wordpress_rating_column'), 999);
            add_filter('comment_row_actions', array($this, 'remove_wordpress_rating_actions'), 999);
        }
    }

    public function add_comment_column($columns)
    {
        if (current_user_can('moderate_comments')) {
            $columns['elyssa_rating'] = __('elyssa AI Rating', 'elyssa-ai');
        }
        return $columns;
    }

    public function comment_column_content($column, $comment_ID)
    {
        if ('elyssa_rating' === $column && current_user_can('moderate_comments')) {
            $rating = get_comment_meta($comment_ID, 'elyssa_ai_rating', true);
            $explanation = get_comment_meta($comment_ID, 'elyssa_ai_explanation', true);

            echo '<div id="elyssa-ai-rating-' . esc_attr($comment_ID) . '">';
            if (!$rating) {
                echo '<div class="elyssa-ai-rating-waiting" title="' . esc_attr__('Waiting for rating', 'elyssa-ai') . '">...</div>';
            } else {
                $rating = $this->get_rating_html($rating, $explanation);
                echo wp_kses_post($rating);
            }
            echo '</div>';
        }
    }

    private function get_rating_html($rating, $explanation)
    {
        $rating = strtolower($rating);
        $settings = [
            'red' => ['#ff0000', __('Inappropriate or spam', 'elyssa-ai')],
            'yellow' => ['#ffff00', __('Questionable', 'elyssa-ai')],
            'green' => ['#00ff00', __('Valuable', 'elyssa-ai')]
        ];

        $color = isset($settings[$rating]) ? $settings[$rating][0] : '#808080';
        $rating_text = isset($settings[$rating]) ? $settings[$rating][1] : __('Unknown', 'elyssa-ai');

        $html = sprintf(
            '<div class="elyssa-ai-rating" style="width: 20px; height: 20px; border-radius: 50%%; background-color: %s;" title="%s"></div><br /><small>%s</small>',
            esc_attr($color),
            esc_attr($rating_text),
            wp_kses_post($explanation)
        );

        return $html;
    }

    public function add_bulk_actions($bulk_actions)
    {
        if (current_user_can('moderate_comments')) {
            $bulk_actions['elyssa_ai_rerate'] = __('Re-evaluate with elyssa AI', 'elyssa-ai');
            $bulk_actions['elyssa_ai_remove_rating'] = __('Remove elyssa AI Rating', 'elyssa-ai');
        }
        return $bulk_actions;
    }

    public function handle_bulk_actions($redirect_to, $doaction, $comment_ids)
    {
        if (!current_user_can('moderate_comments')) {
            return $redirect_to;
        }

        $updated_count = 0;

        if (!elyssa_is_text_enabled()) {
            return $redirect_to;
        }

        if ($doaction === 'elyssa_ai_rerate' || $doaction === 'elyssa_ai_remove_rating') {
            foreach ($comment_ids as $comment_id) {
                if (current_user_can('edit_comment', $comment_id)) {
                    delete_comment_meta($comment_id, 'elyssa_ai_rating');
                    delete_comment_meta($comment_id, 'elyssa_ai_explanation');
                    $updated_count++;
                }
            }
            $redirect_to = add_query_arg($doaction === 'elyssa_ai_rerate' ? 'elyssa_ai_rerated' : 'elyssa_ai_removed', $updated_count, $redirect_to);
        }

        return $redirect_to;
    }

    public function bulk_action_admin_notice()
    {
        if (!empty($_REQUEST['elyssa_ai_rerated'])) {
            $count = intval($_REQUEST['elyssa_ai_rerated']);
            $message = sprintf(
                /* translators: count of selected comments */
                _n(
                    '%s comment has been reset for re-evaluation by elyssa AI.',
                    '%s comments have been reset for re-evaluation by elyssa AI.',
                    $count,
                    'elyssa-ai'
                ),
                number_format_i18n($count)
            );
            echo "<div class='updated'><p>" . esc_html($message) . "</p></div>";
        }

        if (!empty($_REQUEST['elyssa_ai_removed'])) {
            $count = intval($_REQUEST['elyssa_ai_removed']);
            $message = sprintf(
                /* translators: count of selected comments */
                _n(
                    'elyssa AI rating has been removed from %s comment.',
                    'elyssa AI ratings have been removed from %s comments.',
                    $count,
                    'elyssa-ai'
                ),
                number_format_i18n($count)
            );
            echo "<div class='updated'><p>" . esc_html($message) . "</p></div>";
        }
    }

    public function enqueue_scripts($hook)
    {
        if ('edit-comments.php' !== $hook || !current_user_can('moderate_comments')) {
            return;
        }

        wp_enqueue_script('elyssa-ai-comments', plugin_dir_url(__FILE__) . 'assets/elyssa-ai-comments.js?'.time(), array('jquery'), ELYSSA_AI_VERSION, true);
        wp_localize_script('elyssa-ai-comments', 'elyssaAiComments', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('elyssa_ai_rate_comments'),
            'pluginUrl' => plugins_url('elyssa-ai'),
            'rateButtonText' => __('elyssa AI Rate It', 'elyssa-ai'),
            'noCommentsToRate' => __('No comments to rate.', 'elyssa-ai'),
            'ratingInProgress' => __('Rating in progress', 'elyssa-ai'),
            'loadingText' => __('Loading...', 'elyssa-ai'),
            'allCommentsRated' => __('All comments have been rated successfully.', 'elyssa-ai'),
            'someCommentsRated' => __('Rated {success} out of {total} comments successfully. Some comments could not be rated.', 'elyssa-ai'),
            'ratingError' => __('An error occurred during the rating process. Please try again.', 'elyssa-ai'),
            'ratingEnabled' => elyssa_is_text_enabled(),
            'ratingDisabledText' => __('Please enable text services in the settings', 'elyssa-ai')
        ));
    }

    public function ajax_rate_comments()
    {
        check_ajax_referer('elyssa_ai_rate_comments', 'nonce');

        if (!current_user_can('moderate_comments')) {
            wp_send_json_error('Insufficient permissions');
        }

        $comment_id = isset($_POST['comment_id']) ? intval($_POST['comment_id']) : 0;

        if (!$comment_id || !current_user_can('edit_comment', $comment_id)) {
            wp_send_json_error('Invalid comment ID or insufficient permissions');
        }

        $result = $this->rate_comment($comment_id);

        if ($result) {
            $rating = get_comment_meta($comment_id, 'elyssa_ai_rating', true);
            $explanation = get_comment_meta($comment_id, 'elyssa_ai_explanation', true);
            wp_send_json_success(array(
                'html' => $this->get_rating_html($rating, $explanation)
            ));
        } else {
            wp_send_json_error('Rating failed');
        }
    }

    public function rate_new_comment($comment_id, $comment_status)
    {
        if (!get_option('elyssa_ai_comment_rating_enabled', false)) {
            return;
        }

        $this->rate_comment($comment_id);
    }

    private function rate_comment($comment_id)
    {
        if (!get_option('elyssa_ai_comment_rating_enabled', false) || !current_user_can('edit_comment', $comment_id)) {
            return false;
        }

        $init_result = $this->init_text_service();
        if (is_wp_error($init_result)) {
            return false;
        }

        if (!$this->text_service) {
            return false;
        }

        $comment = get_comment($comment_id);
        $content = $comment->comment_content;
        $post_id = $comment->comment_post_ID;
        $post = get_post($post_id);
        $post_content = $post->post_content;
        $post_title = $post->post_title;

        $system_prompt = ServicePrompts::get_system('analyze_comment');
        $user_prompt = sprintf(ServicePrompts::get_user('analyze_comment'), $post_title, substr($post_content, 0, 500), $content);

        try {
            $result = $this->text_service->generate_text($user_prompt, [], $system_prompt);

            $lines = explode("\n", $result);
            $rating = '';
            $explanation = '';
            foreach ($lines as $line) {
                if (strpos($line, 'Color:') === 0) {
                    $rating = strtolower(trim(str_replace('Color:', '', $line)));
                } elseif (strpos($line, 'Reason:') === 0) {
                    $explanation = trim(str_replace('Reason:', '', $line));
                }
            }

            if ($rating && $explanation) {
                update_comment_meta($comment_id, 'elyssa_ai_rating', $rating);
                update_comment_meta($comment_id, 'elyssa_ai_explanation', $explanation);
                return true;
            }
        } catch (Exception $e) {
            elyssa_ai_log(__('Error analyzing comment:', 'elyssa-ai') . ' ' . $e->getMessage());
        }

        return false;
    }

    public function remove_wordpress_rating_column($columns)
    {
        if (isset($columns['comment'])) {
            $columns['comment'] = __('Comment', 'elyssa-ai');
        }
        return $columns;
    }

    public function remove_wordpress_rating_actions($actions)
    {
        unset($actions['spam']);
        unset($actions['unspam']);
        return $actions;
    }
}

// Instantiate the class
new ElyssaAIComments();