<?php
if (!defined('ABSPATH')) {
    exit;
}

class ElyssaAIRequest extends ElyssaAIBaseService
{
    public function __construct()
    {
        add_action('wp_ajax_elyssa_ai_request', array($this, 'handle_ajax_request'));
    }

    public function handle_ajax_request()
    {
        check_ajax_referer('elyssa_ai_nonce', 'nonce');

        $init_result = $this->init_text_service();
        if (is_wp_error($init_result)) {
            wp_send_json_error(array('message' => $init_result->get_error_message()));
            return;
        }

        if (!$this->text_service) {
            wp_send_json_error(array('message' => __('Text AI Service not initialized. Please select a service in the settings.', 'elyssa-ai')));
            return;
        }

        $text = !empty($_POST['text']) ? sanitize_textarea_field(wp_unslash($_POST['text'])) : '';

        $options = array(
            'temperature' => 0.7,
            'max_tokens' => (int)get_option('elyssa_ai_max_tokens', 350),
        );

        $tone_of_voice = !empty($_POST['tone_of_voice']) ? sanitize_text_field(wp_unslash($_POST['tone_of_voice'])) : 'Neutral';
        $article_length = !empty($_POST['length']) ? sanitize_text_field(wp_unslash($_POST['length'])) : 'Short';
        $article_keywords = !empty($_POST['keywords']) ? sanitize_text_field(wp_unslash($_POST['keywords'])) : '';
        $generate_image = !empty($_POST['image']) ? sanitize_text_field(wp_unslash($_POST['image'])) : '';
        $image_prompt = !empty($_POST['image_prompt']) ? sanitize_text_field(wp_unslash($_POST['image_prompt'])) : $text;

        if (empty($text)) {
            wp_send_json_error(array('message' => __('No prompt provided', 'elyssa-ai')));
            return;
        }

        $prompt = $this->generate_prompt($text, $article_keywords, $article_length, $tone_of_voice);

        $system = ServicePrompts::get_system('write_content');

        elyssa_ai_log("Prompt sent to AI: " . $prompt);

        $response = $this->text_service->generate_text($prompt, $options, $system);

        elyssa_ai_log("Raw AI response: " . $response);

        // Entferne eventuell vorhandene Whitespaces am Anfang und Ende
        $response = trim($response);

        // Versuche, die Antwort als JSON zu parsen
        $decoded = parse_structured_content($response);

        // Überprüfe, ob die erwarteten Felder vorhanden sind
        if (!isset($decoded['title']) || !isset($decoded['content']) || !isset($decoded['keywords'])) {
            wp_send_json_error(array('message' => __('Incomplete response from Text AI service.', 'elyssa-ai')));
            return;
        }

        $image = false;
        if ($generate_image == 'on') {
            $image = $this->request_image($image_prompt);
        }

        wp_send_json_success([
            'image' => $image,
            ...$decoded
        ]);
    }

    private function generate_prompt($text, $article_keywords, $article_length, $tone_of_voice)
    {
        return sprintf(ServicePrompts::get_user('write_content'), $text, $article_keywords, $article_length, $tone_of_voice);
    }

    private function get_prompt_from_database($prompt_id)
    {
        $prompt = get_post_meta($prompt_id, '_user_prompt', true);
        if (!$prompt) {
            return new WP_Error('prompt_not_found', __('Selected prompt not found in database.', 'elyssa-ai'));
        }
        return $prompt;
    }

    public function request_image($prompt)
    {
        $init_result = $this->init_image_service();
        if (is_wp_error($init_result)) {
            return $init_result;
        }

        if (!$this->image_service) {
            return new WP_Error('ai_service_error', __('Image AI Service not initialized.', 'elyssa-ai'));
        }

        $response = $this->image_service->generate_image($prompt);

        elyssa_ai_log($response);

        if (is_wp_error($response)) {
            return new WP_Error('ai_service_error', $response->get_error_message());
        } else {
            return $response;
        }
    }
}

// Instantiate the class
new ElyssaAIRequest();