<?php
if (!defined('ABSPATH')) {
    exit;
}

class ElyssaAIDashboard extends ElyssaAIBaseService
{
    public function __construct() {
        add_action('wp_dashboard_setup', array($this, 'add_dashboard_widget'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_dashboard_scripts'));

        add_action('wp_ajax_elyssa_ai_dashboard_generate', array($this, 'handle_dashboard_generate'));
        add_action('wp_ajax_elyssa_ai_create_content', array($this, 'handle_create_content'));
        add_action('wp_ajax_elyssa_ai_dashboard_get_prompt_content', array($this, 'get_prompt_content_ajax_handler'));
    }

    public function add_dashboard_widget() {
        wp_add_dashboard_widget(
            'elyssa_ai_dashboard_widget',
            'elyssa AI Quick Generate',
            array($this, 'dashboard_widget_content')
        );
    }

    public function dashboard_widget_content() {
        $prompts = ElyssaAIPrompts::get_prompts_for_dropdown('text');
        
        // Anstatt direktes HTML auszugeben, erstellen wir ein Container-Element
        echo '<div id="elyssa-ai-dashboard-container"></div>';
        echo '<div id="elyssa-ai-dashboard-loading" style="display: none;"></div>';
        
        // Initialisierung im JavaScript
        wp_enqueue_script('elyssa-ai-dashboard-init-js');
    }

    public function enqueue_dashboard_scripts($hook) {
        if ('index.php' != $hook) {
            return;
        }
        
        // Dashboard-Hauptskript
        wp_enqueue_script(
            'elyssa-ai-dashboard', 
            plugin_dir_url(__FILE__) . 'assets/elyssa-ai-dashboard.js?'.time(), 
            array('jquery'), 
            ELYSSA_AI_VERSION, 
            true
        );
        
        // Hole die Prompts für die JavaScript-Initialisierung
        $prompts = ElyssaAIPrompts::get_prompts_for_dropdown('text');
        $prompt_options = '';
        
        foreach ($prompts as $id => $title) {
            $prompt_options .= '<option value="' . esc_attr($id) . '">' . esc_html($title) . '</option>';
        }
        
        // Initialisierungsskript für das Widget
        wp_register_script(
            'elyssa-ai-dashboard-init-js',
            '',
            array('elyssa-ai-dashboard'),
            ELYSSA_AI_VERSION,
            true
        );
        
        // Inline-Skript zum Erstellen des Widgets
        $init_script = "jQuery(document).ready(function($) {";
        $init_script .= "  const container = $('#elyssa-ai-dashboard-container');";
        $init_script .= "  const loadingDiv = $('#elyssa-ai-dashboard-loading');";
        $init_script .= "  loadingDiv.text('" . esc_js(__('Generating...', 'elyssa-ai')) . "');";
        $init_script .= "  loadingDiv.css({";
        $init_script .= "    'position': 'absolute',";
        $init_script .= "    'top': '0',";
        $init_script .= "    'left': '0',";
        $init_script .= "    'right': '0',";
        $init_script .= "    'bottom': '0',";
        $init_script .= "    'background': 'rgba(255,255,255,0.8)',";
        $init_script .= "    'text-align': 'center',";
        $init_script .= "    'padding-top': '50px'";
        $init_script .= "  });";
        
        $init_script .= "  let widgetHtml = `";
        $init_script .= "    <div id='elyssa-ai-dashboard-widget'>";
        $init_script .= "      <div style='display: flex; align-items: center; margin: 0;'>";
        $init_script .= "        <div style='margin-top: 10px; padding:10px 0px; margin: 0 auto;'>";
        $init_script .= "          <img src='" . esc_js(plugins_url('elyssa-ai/images/elyssa-logo-h100.webp')) . "' alt='elyssa AI Logo' style='height: 60px;'>";
        $init_script .= "        </div>";
        $init_script .= "      </div>";
        
        // Füge die Select-Box hinzu, wenn Prompts vorhanden sind
        if (!empty($prompts)) {
            $init_script .= "      <select id='elyssa-ai-dashboard-prompt-select' style='width: 100%; margin-bottom: 10px;'>";
            $init_script .= "        <option value=''>" . esc_js(__('Select a prompt', 'elyssa-ai')) . "</option>";
            $init_script .= "        $prompt_options";
            $init_script .= "      </select>";
        }
        
        $init_script .= "      <textarea id='elyssa-ai-dashboard-prompt' rows='5' style='width: 100%; margin-bottom: 10px;'";
        $init_script .= "        placeholder='" . esc_js(__('Enter your prompt here', 'elyssa-ai')) . "'></textarea>";
        $init_script .= "      <button id='elyssa-ai-dashboard-generate' class='elyssa-ai-button'";
        $init_script .= "        " . (!elyssa_is_text_enabled() ? "title='" . esc_js(__('Please enable text services in the settings', 'elyssa-ai')) . "' disabled" : "") . "";
        $init_script .= "        style='width: 100%;'>" . esc_js(__('Generate', 'elyssa-ai')) . "</button>";
        $init_script .= "    </div>";
        $init_script .= "  `;
        
        ";
        $init_script .= "  container.html(widgetHtml);";
        $init_script .= "});";        
        
        wp_add_inline_script('elyssa-ai-dashboard-init-js', $init_script);
        
        // Daten für das Dashboard-Script
        wp_localize_script('elyssa-ai-dashboard', 'elyssaAiDashboard', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('elyssa_ai_dashboard_nonce'),
            'prompt_nonce' => wp_create_nonce('elyssa_ai_prompt_nonce'),
        ));
    }

    public function handle_text_generation($prompt)
    {
        $init_result = $this->init_text_service();
        if (is_wp_error($init_result)) {
            return $init_result;
        }

        if (!$this->text_service) {
            return new WP_Error('ai_service_error', __('Text AI Service not initialized. Please select a service in the settings.', 'elyssa-ai'));
        }

        $system_prompt = ServicePrompts::get_system('generate_text');
        $user_prompt = ServicePrompts::get_user('generate_text', $prompt);

        $response = $this->text_service->generate_text($user_prompt, [], $system_prompt);

        if (is_wp_error($response)) {
            return new WP_Error('ai_service_error', __('Error in communication with Text AI service:', 'elyssa-ai') . ' ' . $response->get_error_message());
        }

        $content = parse_structured_content($response);
        if (!$content) {
            return new WP_Error('ai_service_error', __('Invalid response format from Text AI service.', 'elyssa-ai'));
        }

        return $content;
    }

    public function handle_dashboard_generate()
    {
        check_ajax_referer('elyssa_ai_dashboard_nonce', 'nonce');

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'elyssa-ai')));
            return;
        }

        $prompt = isset($_POST['prompt']) ? sanitize_textarea_field(wp_unslash($_POST['prompt'])) : '';

        if (empty($prompt)) {
            wp_send_json_error(array('message' => __('No prompt provided', 'elyssa-ai')));
            return;
        }

        $response = $this->handle_text_generation($prompt);

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
        } else {
            wp_send_json_success(array('content' => $response));
        }
    }

    public function handle_create_content()
    {
        check_ajax_referer('elyssa_ai_dashboard_nonce', 'nonce');

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'elyssa-ai')));
            return;
        }

        $type = isset($_POST['type']) ? sanitize_text_field(wp_unslash($_POST['type'])) : '';
        $content = isset($_POST['content']) ? wp_kses_post(wp_unslash($_POST['content'])) : '';
        $title = isset($_POST['title']) ? wp_kses_post(wp_unslash($_POST['title'])) : '';
        $keywords = isset($_POST['keywords']) ? sanitize_text_field(wp_unslash($_POST['keywords'])) : '';

        if (empty($type) || empty($content)) {
            wp_send_json_error(array('message' => __('Invalid request', 'elyssa-ai')));
            return;
        }

        $post_data = array(
            'post_content' => $content,
            'post_status' => 'draft',
            'post_type' => $type,
            'post_title' => $title
        );

        $post_id = wp_insert_post($post_data);

        if (is_wp_error($post_id)) {
            wp_send_json_error(array('message' => $post_id->get_error_message()));
        } else {
            // Set post tags
            wp_set_post_tags($post_id, explode(',', $keywords));

            $edit_url = get_edit_post_link($post_id, 'raw');
            wp_send_json_success(array('id' => $post_id, 'edit_url' => $edit_url));
        }
    }
}

// Instantiate the class
new ElyssaAIDashboard();