<?php
/**
 * elyssa AI
 *
 * Plugin Name: elyssa AI
 * Plugin URI:  https://elyssaai.com/
 * Description: Revolutionize your WordPress content creation with AI. Generate articles and images, analyze comments, and manage multilingual content using advanced AI services.
 * Version:     2.4
 * Author:      codefabrica
 * Author URI:  https://codefabrica.net
 * License:     GPLv2
 * Text Domain: elyssa-ai
 * Domain Path: /languages
 * Requires PHP: 8.0
 * Tested up to: 6.8
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU
 * General Public License version 2, as published by the Free Software Foundation. You may NOT assume
 * that you can use any other version of the GPL.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

define('ELYSSA_AI_VERSION', '2.4');
define('ELYSSA_AI_PLUGIN_FILE', __FILE__);
define('ELYSSA_AI_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('ELYSSA_AI_PLUGIN_URL', plugin_dir_url(__FILE__));
define('ELYSSA_AI_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Frühere Definitionen für Abwärtskompatibilität
define('ELYSSA_AI_ROOT_PATH', ELYSSA_AI_PLUGIN_DIR);
define('ELYSSA_AI_SITE_URL', get_home_url());

// Ab WordPress 4.6 ist load_plugin_textdomain für WordPress.org-Plugins nicht mehr erforderlich
// Der folgende Code wird nur beibehalten, um WordPress-Versionen vor 4.6 zu unterstützen
if (version_compare($GLOBALS['wp_version'], '4.6', '<')) {
    add_action('plugins_loaded', 'elyssa_ai_load_textdomain');
    
    function elyssa_ai_load_textdomain() {
        load_plugin_textdomain('elyssa-ai', false, dirname(ELYSSA_AI_PLUGIN_BASENAME) . '/languages');
    }
}

require_once ELYSSA_AI_PLUGIN_DIR . 'includes/elyssa-set.php';
require_once ELYSSA_AI_PLUGIN_DIR . 'includes/elyssa-service-prompts.php';
require_once ELYSSA_AI_PLUGIN_DIR . 'includes/elyssa-services.php';
require_once ELYSSA_AI_PLUGIN_DIR . 'includes/elyssa-lib.php';
require_once ELYSSA_AI_PLUGIN_DIR . 'includes/elyssa-logging.php';

require_once ELYSSA_AI_PLUGIN_DIR . 'includes/elyssa-base-service.php';
require_once ELYSSA_AI_PLUGIN_DIR . 'includes/services/class-ai-service.php';
require_once ELYSSA_AI_PLUGIN_DIR . 'includes/services/class-openai-service.php';
require_once ELYSSA_AI_PLUGIN_DIR . 'includes/class-ai-service-factory.php';

add_action('plugins_loaded', function() {
    ElyssaAI_ServiceFactory::register_text_service('openai', 'ElyssaAI_OpenAIService');
    ElyssaAI_ServiceFactory::register_image_service('openai', 'ElyssaAI_OpenAIService');
});

require_once ELYSSA_AI_PLUGIN_DIR . 'admin/elyssa-pages.php';
require_once ELYSSA_AI_PLUGIN_DIR . 'admin/elyssa-settings.php';
require_once ELYSSA_AI_PLUGIN_DIR . 'admin/pages/elyssa-content.php';
require_once ELYSSA_AI_PLUGIN_DIR . 'content/elyssa-request.php';
require_once ELYSSA_AI_PLUGIN_DIR . 'dashboard/elyssa-dashboard.php';
require_once ELYSSA_AI_PLUGIN_DIR . 'media/elyssa-media.php';
require_once ELYSSA_AI_PLUGIN_DIR . 'comments/elyssa-comments.php';
require_once ELYSSA_AI_PLUGIN_DIR . 'editor/elyssa-editor.php';
require_once ELYSSA_AI_PLUGIN_DIR . 'prompts/elyssa-prompts.php';
require_once ELYSSA_AI_PLUGIN_DIR . 'library/elyssa-library.php';
require_once ELYSSA_AI_PLUGIN_DIR . 'bulk/elyssa-bulk.php';
require_once ELYSSA_AI_PLUGIN_DIR . 'chat/elyssa-chat.php';
require_once ELYSSA_AI_PLUGIN_DIR . 'updates/elyssa-updater.php';

add_filter('upload_mimes', function($mimes) {
    $mimes['png'] = 'image/png';
    return $mimes;
});

function elyssa_ai_init() {
    $text_service = get_option('elyssa_ai_text_service', '');
    $image_service = get_option('elyssa_ai_image_service', '');

    if (empty($text_service) && empty($image_service)) {
        add_action('admin_notices', 'elyssa_ai_no_service_notice');
    } else {
        try {
            if (!empty($text_service)) {
                ElyssaAI_ServiceFactory::get_text_service($text_service);
            }
            if (!empty($image_service)) {
                ElyssaAI_ServiceFactory::get_image_service($image_service);
            }
        } catch (Exception $e) {
            add_action('admin_notices', function() use ($e) {
                elyssa_ai_service_error_notice($e->getMessage());
            });
        }
    }
}
add_action('init', 'elyssa_ai_init');

function elyssa_ai_no_service_notice() {
    ?>
    <div class="notice notice-error is-dismissible">
        <p><?php esc_html_e('Elyssa AI: No AI service selected. Please select at least one text or image service in the settings.', 'elyssa-ai'); ?></p>
        <p><a href="<?php echo esc_url(admin_url('admin.php?page=elyssa-ai-settings')); ?>"><?php esc_html_e('To the settings', 'elyssa-ai'); ?></a></p>
    </div>
    <?php
}

function elyssa_ai_service_error_notice($error_message) {
    ?>
    <div class="notice notice-error is-dismissible">
        <p><?php echo sprintf(esc_html__('Elyssa AI: Error when initializing the AI service: %s', 'elyssa-ai'), esc_html($error_message)); ?></p>
    </div>
    <?php
}

function elyssa_ai_plugin_meta($links, $file) {
    if (ELYSSA_AI_PLUGIN_BASENAME == $file) {
        $row_meta = array(
            'docs' => '<a href="' . esc_url('https://elyssaai.com/documentation') . '" target="_blank" aria-label="' . esc_attr__('View elyssa AI documentation', 'elyssa-ai') . '">' . esc_html__('Documentation', 'elyssa-ai') . '</a>'
        );

        return array_merge($links, $row_meta);
    }
    return (array) $links;
}
add_filter('plugin_row_meta', 'elyssa_ai_plugin_meta', 10, 2);

function elyssa_ai_activate() {
    $text_service = get_option('elyssa_ai_text_service');
    $image_service = get_option('elyssa_ai_image_service');

    if (empty($text_service)) {
        update_option('elyssa_ai_text_service', 'openai');
        update_option('elyssa_ai_text_model', 'gpt-4');
    }

    if (empty($image_service)) {
        update_option('elyssa_ai_image_service', 'openai');
        update_option('elyssa_ai_image_model', 'dall-e-2');
    }
    
    // Verschlüssele alle vorhandenen API-Schlüssel
    elyssa_ai_migrate_api_keys();
}

/**
 * Verschlüsselt alle vorhandenen API-Schlüssel bei der Plugin-Aktivierung oder beim Update
 */
function elyssa_ai_migrate_api_keys() {
    // Diese Funktion wird nur einmal bei der Aktivierung oder dem Update aufgerufen
    if (get_option('elyssa_ai_api_keys_encrypted') === 'yes') {
        return;
    }
    
    // Text-API-Schlüssel migrieren
    $text_api_key = get_option('elyssa_ai_text_api_key');
    if (!empty($text_api_key)) {
        // Prüfen, ob der Wert bereits verschlüsselt ist (unwahrscheinlich bei erstmaliger Migration)
        if (strpos($text_api_key, 'sk-') === 0 || strpos($text_api_key, 'bearer') === 0) {
            // Unverschlüsselt - verschlüsseln und speichern
            $encrypted_key = elyssa_ai_encrypt_api_key($text_api_key);
            update_option('elyssa_ai_text_api_key', $encrypted_key);
        }
    }
    
    // Image-API-Schlüssel migrieren
    $image_api_key = get_option('elyssa_ai_image_api_key');
    if (!empty($image_api_key)) {
        // Prüfen, ob der Wert bereits verschlüsselt ist
        if (strpos($image_api_key, 'sk-') === 0 || strpos($image_api_key, 'bearer') === 0) {
            // Unverschlüsselt - verschlüsseln und speichern
            $encrypted_key = elyssa_ai_encrypt_api_key($image_api_key);
            update_option('elyssa_ai_image_api_key', $encrypted_key);
        }
    }
    
    // Markieren, dass die Migration durchgeführt wurde
    update_option('elyssa_ai_api_keys_encrypted', 'yes');
    
    // Logging, wenn aktiviert
    if (get_option('elyssa_ai_logging_enabled')) {
        elyssa_ai_log("API keys encrypted successfully");
    }
}
register_activation_hook(ELYSSA_AI_PLUGIN_FILE, 'elyssa_ai_activate');

// Initialize the plugin updater
function elyssa_ai_init_updater() {
    // Initialize updater
    new ElyssaAI_Updater();
    
    // Handle auto-updates setting
    add_filter('auto_update_plugin', 'elyssa_ai_auto_update', 10, 2);
}

// Register the updater initialization
add_action('admin_init', 'elyssa_ai_init_updater');

/**
 * Manage auto-updates for the plugin
 *
 * @param bool $update Whether to update
 * @param object $item The plugin info
 * @return bool Whether to auto update
 */
function elyssa_ai_auto_update($update, $item) {
    // Make sure this is our plugin
    if (isset($item->slug) && $item->slug === dirname(ELYSSA_AI_PLUGIN_BASENAME)) {
        // Check if auto updates are enabled
        return (bool) get_option('elyssa_ai_auto_update_enabled', true);
    }
    return $update;
}
