<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * Abstract base class for AI services
 *
 * This class defines the structure and required methods for all AI services.
 * Each service must implement these methods and follow the defined standards.
 *
 * @package ElyssaAI
 * @since 2.0.0
 */
abstract class ElyssaAI_Service
{
    /**
     * Get service information and capabilities
     *
     * Required structure:
     * [
     *     'id'          => string   // Unique identifier for the service
     *     'name'        => string   // Display name of the service
     *     'version'     => string   // Service version
     *     'description' => string   // Brief description of the service
     *     'types'       => array    // Supported types ['text', 'image']
     *     'models'      => array    // Available models by type
     *     'features'    => array    // Supported features
     *     'limits'      => array    // Service limitations
     *     'endpoints'   => array    // Optional API endpoints
     * ]
     *
     * @return array Service information and capabilities
     * @since 2.0.0
     */
    abstract public static function get_service_info(): array;

    /**
     * Generate text using the AI service
     *
     * @param string $prompt   The input prompt for text generation
     * @param array  $options  Additional options for generation
     * @param string|null $system System message for the AI
     * @return string|WP_Error Generated text or error
     * @since 2.0.0
     */
    abstract public function generate_text(string $prompt, array $options = [], ?string $system = null);

    /**
     * Generate image using the AI service
     *
     * @param string $prompt  The input prompt for image generation
     * @param array  $options Additional options for generation
     * @return string|WP_Error URL of generated image or error
     * @since 2.0.0
     */
    abstract public function generate_image(string $prompt, array $options = []);

    /**
     * Validate service configuration
     *
     * @return bool|WP_Error True if valid, WP_Error if invalid
     * @since 2.0.0
     */
    abstract public function validate_config();
}
