<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Anthropic Service Implementation
 *
 * Handles text generation using the Anthropic Claude API.
 *
 * @package ElyssaAI
 * @since 2.0.0
 */
class Anthropic_Service extends ElyssaAI_Service
{
    /** @var string API key for Anthropic */
    private $api_key;

    /** @var string Service type (text/image) */
    private $service_type;

    /** @var string Selected AI model */
    private $model;

    /**
     * Initialize the Anthropic service
     *
     * @param string $type Service type (text/image)
     * @param string|null $model Selected model ID
     */
    public function __construct($type, $model = null)
    {
        $this->service_type = $type;
        // API-Schlüssel entschlüsselt abrufen
        $this->api_key = elyssa_ai_get_api_key_option('elyssa_ai_text_api_key');
        $this->model = $model ?? 'claude-3-opus-20240229';
    }

    /**
     * Get service information and capabilities
     *
     * @return array Service information
     */
    public static function get_service_info(): array
    {
        return [
            'id' => 'anthropic',
            'name' => 'Anthropic',
            'version' => '1.0.0',
            'description' => 'Advanced language models from Anthropic',
            'types' => ['text'],

            'features' => [
                'system_messages',
                'function_calling',
                'json_mode',
                'vision',
            ],
            'limits' => [
                'max_tokens' => 4096,
                'max_context_length' => 200000,
            ],
            'endpoints' => [
                'text_completion' => 'https://api.anthropic.com/v1/messages',
            ],
        ];
    }

    /**
     * Generate text using Anthropic Claude
     *
     * @param string $prompt The input prompt for text generation
     * @param array $options Additional options for generation
     * @param string|null $system System message for the AI
     * @return string|WP_Error Generated text or error
     */
    public function generate_text(string $prompt, array $options = [], ?string $system = null): string|WP_Error
    {
        if ($this->service_type !== 'text') {
            return new WP_Error(
                'claude_error',
                esc_html__('This service instance is not configured for text generation', 'elyssa-ai')
            );
        }

        $service_info = self::get_service_info();
        $api_url = $service_info['endpoints']['text_completion'];

        $messages = [];
        if ($system !== null) {
            $messages[] = ['role' => 'system', 'content' => $system];
        }
        $messages[] = ['role' => 'user', 'content' => $prompt];

        $data = [
            'model' => $this->model,
            'messages' => $messages,
            'max_tokens' => $options['max_tokens'] ?? 1000,
            'temperature' => $options['temperature'] ?? 0.7
        ];

        elyssa_ai_log($data);

        $response = wp_remote_post($api_url, [
            'headers' => [
                'x-api-key' => $this->api_key,
                'Content-Type' => 'application/json',
                'anthropic-version' => '2023-06-01'
            ],
            'body' => json_encode($data),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($body['content'][0]['text'])) {
            return $body['content'][0]['text'];
        }

        return new WP_Error(
            'claude_error',
            $body['error']['message'] ?? esc_html__('Unexpected response format from Claude API', 'elyssa-ai')
        );
    }

    /**
     * Generate image using Anthropic
     *
     * @param string $prompt The input prompt for image generation
     * @param array $options Additional options for generation
     * @return string|WP_Error URL of generated image or error
     */
    public function generate_image(string $prompt, array $options = []): string|WP_Error
    {
        return new WP_Error(
            'claude_error',
            esc_html__('Image generation is not supported by Claude', 'elyssa-ai')
        );
    }

    /**
     * Validate service configuration
     *
     * @return bool|WP_Error True if valid, WP_Error if invalid
     */
    public function validate_config(): bool|WP_Error
    {
        if (empty($this->api_key)) {
            return new WP_Error(
                'claude_error',
                esc_html__('Anthropic API key is not configured', 'elyssa-ai')
            );
        }

        if (empty($this->model)) {
            return new WP_Error(
                'claude_error',
                esc_html__('No model specified for Anthropic', 'elyssa-ai')
            );
        }

        return true;
    }
}