<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Flux Service Implementation
 *
 * Handles image generation using the Flux AI API.
 *
 * @package ElyssaAI
 * @since 2.0.0
 */
class Flux_Service extends ElyssaAI_Service
{
    /** @var string API key for Flux */
    private $api_key;

    /** @var string Service type (text/image) */
    private $service_type;

    /** @var string Selected AI model */
    private $model;

    /**
     * Initialize the Flux service
     *
     * @param string $type Service type (text/image)
     * @param string|null $model Selected model ID
     */
    public function __construct($type, $model = null)
    {
        $this->service_type = $type;
        $this->api_key = get_option('elyssa_ai_image_api_key');
        $this->model = $model ?? 'flux-pro-1.1';
    }

    /**
     * Get service information and capabilities
     *
     * @return array Service information
     */
    public static function get_service_info(): array
    {
        return [
            'id' => 'flux',
            'name' => 'Flux AI',
            'version' => '1.0.0',
            'description' => 'Advanced image generation models from Black Forest Labs',
            'types' => ['image'],
            'features' => [
                'high_resolution',
                'fast_inference',
                'size_control',
            ],
            'limits' => [
                'max_tasks' => 24,
                'min_image_size' => 512,
                'max_image_size' => 1024,
            ],
            'endpoints' => [
                'image_generation' => 'https://api.us1.bfl.ai/v1/{model}',
                'result_check' => 'https://api.us1.bfl.ai/v1/get_result',
            ],
        ];
    }

    /**
     * Generate text using Flux
     *
     * @param string $prompt The input prompt for text generation
     * @param array $options Additional options for generation
     * @param string|null $system System message for the AI
     * @return string|WP_Error Generated text or error
     */
    public function generate_text(string $prompt, array $options = [], ?string $system = null): string|WP_Error
    {
        return new WP_Error(
            'flux_error',
            esc_html__('Text generation is not supported by Flux AI', 'elyssa-ai')
        );
    }

    /**
     * Generate image using Flux
     *
     * @param string $prompt The input prompt for image generation
     * @param array $options Additional options for generation
     * @return string|WP_Error URL of generated image or error
     */
    public function generate_image(string $prompt, array $options = []): string|WP_Error
    {
        if ($this->service_type !== 'image') {
            return new WP_Error(
                'flux_error',
                esc_html__('This service instance is not configured for image generation', 'elyssa-ai')
            );
        }

        $api_url = str_replace('{model}', $this->model, self::get_service_info()['endpoints']['image_generation']);

        $data = [
            'prompt' => $prompt,
            'width' => $options['width'] ?? 1024,
            'height' => $options['height'] ?? 768,
        ];

        elyssa_ai_log($data);

        $response = wp_remote_post($api_url, [
            'headers' => [
                'accept' => 'application/json',
                'x-key' => $this->api_key,
                'Content-Type' => 'application/json'
            ],
            'body' => json_encode($data),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($body['id'])) {
            return $this->poll_for_image_result($body['id']);
        }

        return new WP_Error(
            'flux_error',
            $body['error']['message'] ?? esc_html__('Unexpected response format from Flux AI API', 'elyssa-ai')
        );
    }

    /**
     * Poll for image generation result
     *
     * @param string $request_id The request ID to poll for
     * @return string|WP_Error URL of generated image or error
     */
    private function poll_for_image_result(string $request_id): string|WP_Error
    {
        $api_url = self::get_service_info()['endpoints']['result_check'];
        $max_attempts = 60;
        $attempt = 0;

        while ($attempt < $max_attempts) {
            $response = wp_remote_get($api_url, [
                'headers' => [
                    'accept' => 'application/json',
                    'x-key' => $this->api_key,
                ],
                'body' => [
                    'id' => $request_id,
                ],
            ]);

            if (is_wp_error($response)) {
                return $response;
            }

            $body = json_decode(wp_remote_retrieve_body($response), true);

            if (isset($body['status']) && $body['status'] === 'Ready' && isset($body['result']['sample'])) {
                return $body['result']['sample'];
            }

            if (isset($body['status']) && $body['status'] === 'Failed') {
                return new WP_Error(
                    'flux_error',
                    esc_html__('Image generation failed', 'elyssa-ai')
                );
            }

            $attempt++;
            sleep(0.5);
        }

        return new WP_Error(
            'flux_error',
            esc_html__('Image generation timed out', 'elyssa-ai')
        );
    }

    /**
     * Validate service configuration
     *
     * @return bool|WP_Error True if valid, WP_Error if invalid
     */
    public function validate_config(): bool|WP_Error
    {
        if (empty($this->api_key)) {
            return new WP_Error(
                'flux_error',
                esc_html__('Flux API key is not configured', 'elyssa-ai')
            );
        }

        $models = self::get_service_info()['models'][$this->service_type];
        if (!isset($models[$this->model])) {
            return new WP_Error(
                'flux_error',
                esc_html__('Invalid model selected for Flux AI', 'elyssa-ai')
            );
        }

        return true;
    }
} 