<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Leonardo Service Implementation
 *
 * Handles image generation using the Leonardo AI API.
 *
 * @package ElyssaAI
 * @since 2.0.0
 */
class Leonardo_Service extends ElyssaAI_Service
{
    /** @var string API key for Leonardo */
    private $api_key;

    /** @var string Service type (text/image) */
    private $service_type;

    /** @var string Selected AI model */
    private $model;

    /**
     * Initialize the Leonardo service
     *
     * @param string $type Service type (text/image)
     * @param string|null $model Selected model ID
     */
    public function __construct($type, $model = null)
    {
        $this->service_type = $type;
        // API-Schlüssel entschlüsselt abrufen
        $this->api_key = elyssa_ai_get_api_key_option('elyssa_ai_image_api_key');
        $this->model = $model ?? 'phoenix-v1';
    }

    /**
     * Get service information and capabilities
     *
     * @return array Service information
     */
    public static function get_service_info(): array
    {
        return [
            'id' => 'leonardo',
            'name' => 'Leonardo AI',
            'version' => '1.0.0',
            'description' => 'Advanced image generation models from Leonardo AI',
            'types' => ['image'],
            'features' => [
                'negative_prompts',
                'seed_control',
                'size_control',
                'prompt_magic',
                'style_transfer',
            ],
            'limits' => [
                'max_image_size' => 1024,
                'min_image_size' => 512,
                'max_samples' => 4,
            ],
            'endpoints' => [
                'image_generation' => 'https://cloud.leonardo.ai/api/rest/v1/generations',
                'generation_status' => 'https://cloud.leonardo.ai/api/rest/v1/generations/{id}',
            ],
        ];
    }

    /**
     * Generate text using Leonardo
     *
     * @param string $prompt The input prompt for text generation
     * @param array $options Additional options for generation
     * @param string|null $system System message for the AI
     * @return string|WP_Error Generated text or error
     */
    public function generate_text(string $prompt, array $options = [], ?string $system = null): string|WP_Error
    {
        return new WP_Error(
            'leonardo_error',
            esc_html__('Text generation is not supported by Leonardo AI', 'elyssa-ai')
        );
    }

    /**
     * Generate image using Leonardo
     *
     * @param string $prompt The input prompt for image generation
     * @param array $options Additional options for generation
     * @return string|WP_Error URL of generated image or error
     */
    public function generate_image(string $prompt, array $options = []): string|WP_Error
    {
        if ($this->service_type !== 'image') {
            return new WP_Error(
                'leonardo_error',
                esc_html__('This service instance is not configured for image generation', 'elyssa-ai')
            );
        }

        $service_info = self::get_service_info();
        $api_url = $service_info['endpoints']['image_generation'];

        $data = [
            'prompt' => $prompt,
            'modelId' => $this->model,
            'width' => $options['width'] ?? 1024,
            'height' => $options['height'] ?? 1024,
            'num_images' => $options['num_images'] ?? 1,
            'negative_prompt' => $options['negative_prompt'] ?? '',
            'promptMagic' => $options['prompt_magic'] ?? false,
        ];

        elyssa_ai_log($data);

        $response = wp_remote_post($api_url, [
            'headers' => [
                'Accept' => 'application/json',
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'application/json'
            ],
            'body' => json_encode($data),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($body['sdGenerationJob']['generationId'])) {
            return $this->poll_for_image_result($body['sdGenerationJob']['generationId']);
        }

        return new WP_Error(
            'leonardo_error',
            $body['error']['message'] ?? esc_html__('Unexpected response format from Leonardo AI API', 'elyssa-ai')
        );
    }

    /**
     * Poll for image generation result
     *
     * @param string $generation_id The generation ID to poll for
     * @return string|WP_Error URL of generated image or error
     */
    private function poll_for_image_result(string $generation_id): string|WP_Error
    {
        $service_info = self::get_service_info();
        $api_url = str_replace('{id}', $generation_id, $service_info['endpoints']['generation_status']);
        $max_attempts = 30;
        $attempt = 0;

        while ($attempt < $max_attempts) {
            $response = wp_remote_get($api_url, [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->api_key,
                    'Accept' => 'application/json',
                ],
            ]);

            if (is_wp_error($response)) {
                return $response;
            }

            $body = json_decode(wp_remote_retrieve_body($response), true);

            if (isset($body['generations_by_pk']['generated_images'][0]['url'])) {
                return $body['generations_by_pk']['generated_images'][0]['url'];
            }

            if (isset($body['generations_by_pk']['status']) && $body['generations_by_pk']['status'] === 'FAILED') {
                return new WP_Error(
                    'leonardo_error',
                    esc_html__('Image generation failed', 'elyssa-ai')
                );
            }

            $attempt++;
            sleep(2);
        }

        return new WP_Error(
            'leonardo_error',
            esc_html__('Image generation timed out', 'elyssa-ai')
        );
    }

    /**
     * Validate service configuration
     *
     * @return bool|WP_Error True if valid, WP_Error if invalid
     */
    public function validate_config(): bool|WP_Error
    {
        if (empty($this->api_key)) {
            return new WP_Error(
                'leonardo_error',
                esc_html__('Leonardo API key is not configured', 'elyssa-ai')
            );
        }

        $models = self::get_service_info()['models'][$this->service_type];
        if (!isset($models[$this->model])) {
            return new WP_Error(
                'leonardo_error',
                esc_html__('Invalid model selected for Leonardo AI', 'elyssa-ai')
            );
        }

        return true;
    }
}