<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Mistral Service Implementation
 *
 * Handles text generation using the Mistral AI API.
 *
 * @package ElyssaAI
 * @since 2.0.0
 */
class Mistral_Service extends ElyssaAI_Service
{
    /** @var string API key for Mistral */
    private $api_key;

    /** @var string Service type (text/image) */
    private $service_type;

    /** @var string Selected AI model */
    private $model;

    /**
     * Initialize the Mistral service
     *
     * @param string $type Service type (text/image)
     * @param string|null $model Selected model ID
     */
    public function __construct($type, $model = null)
    {
        $this->service_type = $type;
        $this->api_key = get_option('elyssa_ai_text_api_key');
        $this->model = $model ?? 'mistral-large-latest';
    }

    /**
     * Get service information and capabilities
     *
     * @return array Service information
     */
    public static function get_service_info(): array
    {
        return [
            'id' => 'mistral',
            'name' => 'Mistral AI',
            'version' => '1.0.0',
            'description' => 'Advanced language models from Mistral AI',
            'types' => ['text'],
            'features' => [
                'system_messages',
                'function_calling',
                'json_mode',
            ],
            'limits' => [
                'max_tokens' => 4096,
                'max_context_length' => 32768,
            ],
            'endpoints' => [
                'text_completion' => 'https://api.mistral.ai/v1/chat/completions',
            ],
        ];
    }

    /**
     * Generate text using Mistral
     *
     * @param string $prompt The input prompt for text generation
     * @param array $options Additional options for generation
     * @param string|null $system System message for the AI
     * @return string|WP_Error Generated text or error
     */
    public function generate_text(string $prompt, array $options = [], ?string $system = null): string|WP_Error
    {
        if ($this->service_type !== 'text') {
            return new WP_Error(
                'mistral_error',
                esc_html__('This service instance is not configured for text generation', 'elyssa-ai')
            );
        }

        $service_info = self::get_service_info();
        $api_url = $service_info['endpoints']['text_completion'];

        $messages = [];
        if ($system !== null) {
            $messages[] = ['role' => 'system', 'content' => $system];
        }
        $messages[] = ['role' => 'user', 'content' => $prompt];

        $data = [
            'model' => $this->model,
            'messages' => $messages,
            'temperature' => $options['temperature'] ?? 0.7,
            'max_tokens' => $options['max_tokens'] ?? 1000
        ];

        elyssa_ai_log($data);

        $response = wp_remote_post($api_url, [
            'headers' => [
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ],
            'body' => json_encode($data),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($body['choices'][0]['message']['content'])) {
            return $body['choices'][0]['message']['content'];
        }

        return new WP_Error(
            'mistral_error',
            $body['error']['message'] ?? esc_html__('Unexpected response format from Mistral AI API', 'elyssa-ai')
        );
    }

    /**
     * Generate image using Mistral
     *
     * @param string $prompt The input prompt for image generation
     * @param array $options Additional options for generation
     * @return string|WP_Error URL of generated image or error
     */
    public function generate_image(string $prompt, array $options = []): string|WP_Error
    {
        return new WP_Error(
            'mistral_error',
            esc_html__('Image generation is not supported by Mistral AI', 'elyssa-ai')
        );
    }

    /**
     * Validate service configuration
     *
     * @return bool|WP_Error True if valid, WP_Error if invalid
     */
    public function validate_config(): bool|WP_Error
    {
        if (empty($this->api_key)) {
            return new WP_Error(
                'mistral_error',
                esc_html__('Mistral API key is not configured', 'elyssa-ai')
            );
        }

        $models = self::get_service_info()['models'][$this->service_type];
        if (!isset($models[$this->model])) {
            return new WP_Error(
                'mistral_error',
                esc_html__('Invalid model selected for Mistral AI', 'elyssa-ai')
            );
        }

        return true;
    }
}