<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * Handles plugin updates for Elyssa AI
 * 
 * Communicates with the update server to check for and download new plugin versions
 * 
 * @package ElyssaAI
 * @since 2.1.0
 */
class ElyssaAI_Updater {
    /**
     * Update server URL
     *
     * @var string
     */
    private $update_url;

    /**
     * Current plugin version
     *
     * @var string
     */
    private $current_version;

    /**
     * Plugin basename
     *
     * @var string
     */
    private $plugin_basename;

    /**
     * Plugin slug
     *
     * @var string
     */
    private $plugin_slug;

    /**
     * Initialize the updater
     */
    public function __construct() {
        $this->update_url = 'https://get.codefabrica.net/wp/elyssa-ai/info';
        $this->current_version = ELYSSA_AI_VERSION;
        $this->plugin_basename = ELYSSA_AI_PLUGIN_BASENAME;
        $this->plugin_slug = dirname($this->plugin_basename);

        // Add filters for the update process
        add_filter('pre_set_site_transient_update_plugins', array($this, 'check_for_update'));
        add_filter('plugins_api', array($this, 'plugin_info'), 10, 3);
        add_filter('upgrader_post_install', array($this, 'after_install'), 10, 3);
        add_filter('site_transient_update_plugins', array($this, 'ensure_update_format'));
        
        // Add a custom message on the plugin screen
        add_action('in_plugin_update_message-' . $this->plugin_basename, array($this, 'update_message'), 10, 2);
        
        // Register AJAX handler for manual update check
        add_action('wp_ajax_elyssa_ai_check_update', array($this, 'ajax_check_update'));
        add_action('wp_ajax_elyssa_ai_reset_transients', array($this, 'ajax_reset_transients'));

        // Pre-check für die Admin-Seiten
        add_action('admin_init', array($this, 'maybe_reset_update_transients'), 5);
    }

    /**
     * Check for plugin updates
     *
     * @param object $transient The update transient object
     * @return object The modified transient
     */
    public function check_for_update($transient) {
        if (empty($transient->checked)) {
            return $transient;
        }

        // Immer unsere Plugin-Version definieren, auch wenn nicht explizit in checked
        $transient->checked[$this->plugin_basename] = $this->current_version;
        
        // Get update info
        $update_info = $this->request_update_info();
        
        // Check if the API call returned an error
        if (is_wp_error($update_info)) {
            if (get_option('elyssa_ai_logging_enabled')) {
                elyssa_ai_log("Update check error: " . $update_info->get_error_message());
            }
            return $transient;
        }

        // If a newer version is available, add it to the transient
        if (isset($update_info->version) && version_compare($this->current_version, $update_info->version, '<')) {
            // Create plugin update data
            $plugin_data = new stdClass();
            $plugin_data->slug = $this->plugin_slug;
            $plugin_data->plugin = $this->plugin_basename;
            $plugin_data->new_version = $update_info->version;
            $plugin_data->url = isset($update_info->homepage) ? $update_info->homepage : '';
            $plugin_data->package = isset($update_info->download_link) ? $update_info->download_link : '';
            
            // Additional information
            if (isset($update_info->tested)) {
                $plugin_data->tested = $update_info->tested;
            }
            if (isset($update_info->requires_php)) {
                $plugin_data->requires_php = $update_info->requires_php;
            }
            
            // Make sure compatibility is an object
            $plugin_data->compatibility = new stdClass();
            
            // Set the response
            $transient->response[$this->plugin_basename] = $plugin_data;
            
            if (get_option('elyssa_ai_logging_enabled')) {
                elyssa_ai_log("Update check: New version {$update_info->version} available");
            }
        } else {
            // Wenn keine neuere Version verfügbar ist, stellen wir sicher, dass keine alte Update-Nachricht angezeigt wird
            if (isset($transient->response[$this->plugin_basename])) {
                unset($transient->response[$this->plugin_basename]);
            }
            
            if (get_option('elyssa_ai_logging_enabled')) {
                elyssa_ai_log("Update check: Current version is up to date");
            }
        }
        
        return $transient;
    }

    /**
     * Request update information from the server
     *
     * @return object|WP_Error The update information or error
     */
    private function request_update_info() {
        // Build the URL with query parameters
        $params = array(
            'action' => 'plugin_information',
            'slug' => $this->plugin_slug,
            'version' => $this->current_version
        );
        
        $request_url = add_query_arg($params, $this->update_url);
        
        // SSL-Verifizierung für Testzwecke deaktivieren
        $args = array(
            'timeout' => 15,
            'sslverify' => false // SSL-Verifizierung deaktivieren
        );
        
        // Make the request
        $response = wp_remote_get($request_url, $args);
        
        // Check for errors
        if (is_wp_error($response)) {
            return $response;
        }
        
        // Get response code
        $response_code = wp_remote_retrieve_response_code($response);
        
        if ($response_code != 200) {
            return new WP_Error('non_200_response', 'Server returned response code: ' . $response_code);
        }

        // Get and parse the body
        $body = wp_remote_retrieve_body($response);
        
        // Try to decode the JSON
        $data = json_decode($body);
        
        // Check for JSON errors
        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error('json_decode_error', 'Failed to parse JSON: ' . json_last_error_msg());
        }

        if (empty($body) || !is_object($data)) {
            return new WP_Error('invalid_response', 'Invalid update server response');
        }

        return $data;
    }

    /**
     * Provide detailed plugin information for the update screen
     *
     * @param object|false $result The result object
     * @param string $action The API action being performed
     * @param object $args Arguments for the API call
     * @return object|false The result or false
     */
    public function plugin_info($result, $action, $args) {
        // Make sure we're getting info about our plugin
        if ('plugin_information' != $action || !isset($args->slug) || $args->slug != $this->plugin_slug) {
            return $result;
        }
        
        // Get plugin info
        $plugin_info = $this->request_update_info();

        if (is_wp_error($plugin_info)) {
            return $result;
        }

        if (is_object($plugin_info) && !empty($plugin_info)) {
            // Fix for WordPress expecting certain properties to be arrays
            if (isset($plugin_info->sections) && is_object($plugin_info->sections)) {
                $plugin_info->sections = (array) $plugin_info->sections;
            }
            
            if (isset($plugin_info->banners) && is_object($plugin_info->banners)) {
                $plugin_info->banners = (array) $plugin_info->banners;
            }
            
            if (isset($plugin_info->icons) && is_object($plugin_info->icons)) {
                $plugin_info->icons = (array) $plugin_info->icons;
            }
            
            if (isset($plugin_info->contributors) && is_object($plugin_info->contributors)) {
                $plugin_info->contributors = (array) $plugin_info->contributors;
            }
            
            if (isset($plugin_info->compatibility) && is_object($plugin_info->compatibility)) {
                $plugin_info->compatibility = (array) $plugin_info->compatibility;
            }
            
            return $plugin_info;
        }

        return $result;
    }

    /**
     * Actions to perform after a plugin update
     *
     * @param bool $response Installation response
     * @param array $hook_extra Extra installation arguments
     * @param array $result Installation result data
     * @return array The result data
     */
    public function after_install($response, $hook_extra, $result) {
        if (isset($hook_extra['plugin']) && $hook_extra['plugin'] == $this->plugin_basename) {
            // Clear update cache after successful install
            delete_transient('elyssa_ai_update_check');
            delete_transient('elyssa_ai_plugin_info');
            delete_site_transient('update_plugins');
            
            // API-Schlüssel nach Update verschlüsseln
            if (function_exists('elyssa_ai_migrate_api_keys')) {
                elyssa_ai_migrate_api_keys();
                if (get_option('elyssa_ai_logging_enabled')) {
                    elyssa_ai_log("API keys encryption check performed after plugin update");
                }
            }
        }
        
        return $result;
    }
    
    /**
     * Force check for plugin updates
     *
     * @return object|WP_Error Update information or error
     */
    public function force_update_check() {
        // Clear existing update cache
        delete_transient('elyssa_ai_update_check');
        delete_transient('elyssa_ai_plugin_info');
        delete_site_transient('update_plugins');
        
        // Get update info
        $update_info = $this->request_update_info();
        
        // If we have update info and there's a newer version, add it to the transient
        if (!is_wp_error($update_info) && isset($update_info->version)) {
            if (version_compare($this->current_version, $update_info->version, '<')) {
                // Log the result if logging is enabled
                if (get_option('elyssa_ai_logging_enabled')) {
                    elyssa_ai_log("Manual update check: New version {$update_info->version} available");
                }
                
                return $update_info;
            } else {
                // Log the result if logging is enabled
                if (get_option('elyssa_ai_logging_enabled')) {
                    elyssa_ai_log("Manual update check: Current version is up to date");
                }
                
                return new WP_Error('no_update', __('You have the latest version of Elyssa AI installed.', 'elyssa-ai'));
            }
        }
        
        // Log any errors
        if (is_wp_error($update_info) && get_option('elyssa_ai_logging_enabled')) {
            elyssa_ai_log("Manual update check failed: " . $update_info->get_error_message());
        }
        
        return $update_info;
    }

    /**
     * Display custom update messages on the plugin screen
     *
     * @param array $plugin_data An array of plugin metadata
     * @param object $response The update response object
     */
    public function update_message($plugin_data, $response) {
        if (empty($response->package)) {
            echo ' ' . sprintf(
                __('Note: This update might require a valid license. Please visit %s for more information.', 'elyssa-ai'),
                '<a href="https://elyssaai.com" target="_blank">elyssaai.com</a>'
            );
        }
    }
    
    /**
     * Handle AJAX requests for manual update checks
     */
    public function ajax_check_update() {
        // Überprüfe die Nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'elyssa_ai_check_update_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'elyssa-ai')));
        }
        
        // Überprüfe Berechtigungen
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission to perform this action.', 'elyssa-ai')));
        }
        
        // Führe den Update-Check durch
        $update_info = $this->force_update_check();
        
        // Verarbeite die Rückgabe
        if (is_wp_error($update_info)) {
            wp_send_json_error(array('message' => $update_info->get_error_message()));
        } else if (isset($update_info->version)) {
            // Prüfe, ob eine neuere Version verfügbar ist
            if (version_compare($this->current_version, $update_info->version, '<')) {
                // Neue Version verfügbar
                wp_send_json_success(array(
                    'version' => $update_info->version,
                    'current_version' => $this->current_version,
                    'download_link' => isset($update_info->download_link) ? $update_info->download_link : '',
                    'update_available' => true
                ));
            } else {
                // Aktuelle Version ist auf dem neuesten Stand
                wp_send_json_success(array(
                    'version' => $this->current_version,
                    'current_version' => $this->current_version,
                    'update_available' => false
                ));
            }
        } else {
            // Unerwartetes Format
            wp_send_json_error(array('message' => __('Received invalid update information from the server.', 'elyssa-ai')));
        }
    }
    
    /**
     * Handle AJAX requests for resetting transients
     */
    public function ajax_reset_transients() {
        // Überprüfe die Nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'elyssa_ai_check_update_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'elyssa-ai')));
        }
        
        // Überprüfe Berechtigungen
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission to perform this action.', 'elyssa-ai')));
        }
        
        // Transients zurücksetzen
        $this->reset_all_update_caches();
        
        // Erneutes Update-Check erzwingen
        wp_update_plugins();
        
        wp_send_json_success(array('message' => __('Transients have been reset successfully.', 'elyssa-ai')));
    }
    
    /**
     * Reset all update caches and transients
     */
    public function reset_all_update_caches() {
        // WordPress-eigene Transients löschen
        delete_site_transient('update_plugins');
        delete_site_transient('update_themes');
        delete_site_transient('update_core');
        
        // Plugin-spezifische Transients löschen
        delete_transient('elyssa_ai_update_check');
        delete_transient('elyssa_ai_plugin_info');
        delete_transient('elyssa_ai_last_update_reset');
        
        // Cache bereinigen
        wp_clean_plugins_cache(true);
        
        // Logging, wenn aktiviert
        if (get_option('elyssa_ai_logging_enabled')) {
            elyssa_ai_log("All update caches have been reset and refreshed");
        }
        
        return true;
    }
    
    /**
     * Ensure the update format is correct
     * 
     * This is a final check to make sure the update data is correctly formatted
     *
     * @param object $transient The update transient
     * @return object The modified transient
     */
    public function ensure_update_format($transient) {
        if (empty($transient) || empty($transient->response)) {
            return $transient;
        }
        
        // Stelle sicher, dass das Update für unser Plugin korrekt formatiert ist
        if (isset($transient->response[$this->plugin_basename])) {
            $plugin_data = $transient->response[$this->plugin_basename];
            
            // Stelle sicher, dass die wichtigsten Felder vorhanden sind
            if (!isset($plugin_data->compatibility) || !is_object($plugin_data->compatibility)) {
                $plugin_data->compatibility = new stdClass();
            }
            
            // Diese Felder müssen gesetzt sein, damit WordPress das Update erkennt
            if (!isset($plugin_data->slug)) {
                $plugin_data->slug = $this->plugin_slug;
            }
            
            if (!isset($plugin_data->plugin)) {
                $plugin_data->plugin = $this->plugin_basename;
            }
            
            // Update-Daten zurückschreiben
            $transient->response[$this->plugin_basename] = $plugin_data;
            
            // Logging für Debugging
            if (get_option('elyssa_ai_logging_enabled')) {
                elyssa_ai_log("Update format has been corrected for {$this->plugin_basename}");
            }
        }
        
        return $transient;
    }
    
    /**
     * Prüft, ob die Update-Transients zurückgesetzt werden sollten
     */
    public function maybe_reset_update_transients() {
        // Nur für Admin-Seiten ausführen
        if (!is_admin()) {
            return;
        }
        
        // Prüfe, ob wir uns auf der Plugin-Seite oder der Update-Seite befinden
        $current_screen = function_exists('get_current_screen') ? get_current_screen() : null;
        if (!$current_screen || !in_array($current_screen->base, array('plugins', 'update-core', 'update'))) {
            return;
        }
        
        // Nur alle 12 Stunden ausführen (um zu viele API-Anfragen zu vermeiden)
        $last_reset = get_transient('elyssa_ai_last_update_reset');
        if ($last_reset) {
            return;
        }
        
        // Caches zurücksetzen
        $this->reset_all_update_caches();
        
        // Setze Transient für 12 Stunden
        set_transient('elyssa_ai_last_update_reset', time(), 12 * HOUR_IN_SECONDS);
    }
}
